/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var Long = require('long'),
    IntKeyLinkedMap = require('../util/intkey-linkedmap'),
    ValueEnum = require('./valueenum'),
    Value = require('./value'),
    BooleanValue = require('./boolean-value'),
    NumberValue = require('./number-value'),
    TextValue = require('./text-value'),
    DecimalValue = require('./decimal-value'),
    ListValue = require('./list-value');

function IntMapValue() {
    Value.call(this);

    this.table = new IntKeyLinkedMap();
    this.equals = function (obj) {
        return (this === obj) 
    };
}

IntMapValue.prototype = new Value();
IntMapValue.prototype.constructor = IntMapValue;
IntMapValue.prototype.size = function () {
    return this.table.size();
};
IntMapValue.prototype.isEmpty = function () {
    return this.size() === 0;
};
IntMapValue.prototype.containsKey = function (key) {
    return this.table.containsKey(key);
};
IntMapValue.prototype.keys = function () {
    return this.table.keys();
};
IntMapValue.prototype.get = function (key) {
    return this.table.get(key);
};
IntMapValue.prototype.getBoolean = function (key) {
    var v = this.get(key);
    if (v instanceof BooleanValue) {
        return v.value;
    }
    return false;
};
IntMapValue.prototype.getInt = function (key) {
    var v = this.get(key);
    if (v instanceof NumberValue) {
        return v.intValue();
    }
    return 0;
};
IntMapValue.prototype.getLong = function (key) {
    var v = this.get(key);
    if (v instanceof NumberValue) {
        return v.longValue();
    }
    return Long.ZERO;
};
IntMapValue.prototype.getNumber = function (key) {
    var v = this.get(key);
    if (v instanceof NumberValue) {
        return v.numberValue();
    }
    return 0;
};
IntMapValue.prototype.getFloat = function (key) {
    var v = this.get(key);
    if (v instanceof NumberValue) {
        return v.floatValue();
    }
    return 0;

};
IntMapValue.prototype.getText = function (key) {
    var v = this.get(key);
    if (v instanceof TextValue) {
        return v.value;
    }
    return null;
};
IntMapValue.prototype.putValue = function (key, value) {
    this.table.put(key,value);
    return this;
};
IntMapValue.prototype.putString = function (key, value) {
    this.table.put(key, new TextValue(value + ""));
    return this;
};
IntMapValue.prototype.putLong = function (key, value) {
    this.table.put(key, new DecimalValue(value));
};
IntMapValue.prototype.remove = function (key) {
    return this.table.remove(key);
};
IntMapValue.prototype.clear = function () {
    this.table.clear();
};
IntMapValue.prototype.toString = function () {
    return this.table.toString();
};
IntMapValue.prototype.getValueType = function () {
    return ValueEnum.INT_MAP;
};
IntMapValue.prototype.write = function (dout) {
    dout.writeDecimal(this.table.size());
    var en = this.table.keys();  
    while(en.hasMoreElements()) {
        var key = en.nextElement();
        dout.writeInt32BE(key);
        dout.writeValue(this.table.get(key));
    }
};
IntMapValue.prototype.read = function (din) {
    var count = din.readDecNumber();
    for (var i = 0; i < count; i++) {
        var key = din.readInt32BE();
        var value = din.readValue();
        this.putValue(key, value);
    }
    return this;
};
IntMapValue.prototype.newList = function (key) {
    var list = new ListValue();
    this.putValue(key, list);
    return list;
};
IntMapValue.prototype.getList = function (key) {
    return this.table.get(key);
};
IntMapValue.prototype.toObject = function () {
    return this.table;
};
IntMapValue.prototype.putAllMap = function (m) {
    var en = m.keys();
    while (en.hasMoreElements()) {
        var key = en.nextElement();
        var value = m.get(key);
        if (value instanceof Value) {
            this.table.put(key, value);
        }
    }
};
IntMapValue.prototype.putAllMapValue = function (m) {
    this.putAllMap(m);
};

module.exports = IntMapValue;