/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var Value        = require('./value'),
    ValueEnum    = require('./valueenum'),
    BooleanValue = require('./boolean-value'),
    NumberValue  = require('./number-value'),
    DecimalValue = require('./decimal-value'),
    FloatValue   = require('./float-value'),
    DoubleValue  = require('./double-value'),
    TextValue    = require('./text-value'),
    Long         = require('long');

function ListValue(valueList) {
    Value.call(this);
    this.valueList = valueList || [];
    this.getBoolean = function(i) {
        var v = this.valueList[i];
        if(v instanceof BooleanValue) {
            return v.value;
        }
        return false;
    };
    this.getDouble = function(i) {
        var v = this.valueList[i];
        if(v.doubleValue) {
            return v.doubleValue();
        }
        return 0.0;
    };
    this.getLong = function(i) {
        var v = this.valueList[i];
        if(v.longValue) {
            return v.longValue();
        }
        return Long.ZERO;
    };
    this.getNumber = function(i) {
        var v = this.valueList[i];
        if(v.numberValue) {
            return v.numberValue();
        }
        return 0;
    };
    this.getInt = function(i) {
        var v = this.valueList[i];
        if(v instanceof NumberValue) {
            return v.intValue();
        }
        return 0;
    };
    this.getString = function(i) {
        var v = this.valueList[i];
        if(v instanceof TextValue) {
            return v.value;
        }
        if(v == null) {
            return null;
        }
        return v.toString();
    };
    this.set = function(i, value) {
        this.valueList[i] = value;
    };
    this.addValue = function(value) {
        this.valueList.push(value);
        return this;
    };
    this.addBoolean = function(value) {
        this.valueList.push(new BooleanValue(value));
        return this;
    };
    this.addDouble = function(value) {
        this.valueList.push(new DoubleValue(value));
        return this;
    };
    this.addLong = function(value) {
        this.valueList.push(new DecimalValue(value));
        return this;
    };
    this.addFloat = function(value) {
        this.valueList.push(new FloatValue(value));
        return this;
    };
    this.addString = function(value) {
        this.valueList.push(new TextValue(value));
        return this;
    };
    this.addStringArray = function(value) {
        if(Array.isArray(value)) {
            var len = value.length;
            for(var i=0; i<len; i++) {
                this.addString(value[i]);
            }
        }
        return this;
    };
    this.addValueArray = function(value) {
        if(Array.isArray(value)) {
            var len = value.length;
            for(var i=0; i<len; i++) {
                this.addValue(value[i]);
            }
        };
    };
    this.size = function() {
        return this.valueList.length;
    };
}

ListValue.prototype = new Value();
ListValue.prototype.constructor = ListValue;
ListValue.prototype.write = function(dout) {
    var sz = this.size();
    dout.writeDecimal(sz);
    for(var i=0; i<sz; i++) {
        dout.writeValue(this.valueList[i]);
    }
};
ListValue.prototype.read = function(din) {
    var count = din.readDecNumber();
    for(var i=0; i<count; i++) {
        this.addValue(din.readValue());
    }
    return this;
};
ListValue.prototype.getValueType = function () {
    return ValueEnum.LIST;
};
ListValue.prototype.toObject = function() {
    return this.valueList;
};

module.exports = ListValue;