/**
 * PacketQueue - Queue implementation for async sending packets
 * Implemented as a singleton to ensure only one queue exists.
 */

class PacketQueue {
    constructor(maxSize = 1000) {
        // Singleton pattern: If instance exists, return it
        if (PacketQueue.instance) {
            return PacketQueue.instance;
        }

        // Initialize queue
        this.queue = [];
        this.maxSize = maxSize;

        // Store the instance
        PacketQueue.instance = this;
    }

    put(item) {
        if (this.queue.length < this.maxSize) {
            this.queue.push(item);
            return true;
        }
        return false;
    }

    get() {
        if (this.queue.length > 0) {
            return this.queue.shift();
        }
        return null;
    }

    full() {
        return this.queue.length >= this.maxSize;
    }

    empty() {
        return this.queue.length === 0;
    }

    size() {
        return this.queue.length;
    }

    clear() {
        this.queue = [];
    }

    /**
     * Get the singleton instance
     * @param {number} maxSize - Maximum size of the queue (only used if instance doesn't exist)
     * @returns {PacketQueue} - The singleton instance
     */
    static getInstance(maxSize = 1000) {
        if (!PacketQueue.instance) {
            PacketQueue.instance = new PacketQueue(maxSize);
        }
        return PacketQueue.instance;
    }
}

// Initialize the static instance property
PacketQueue.instance = null;

// Export the class
module.exports = PacketQueue;