/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var TraceContextManager = require('../trace/trace-context-manager');

const Hexa32 = require("../util/hexa32");
const SecurityMaster = require("../net/security-master");
const conf = require("../conf/configure");
const KeyGen = require("../util/keygen");
const TraceHelper = require('../util/trace-helper');
const shimmer = require('../core/shimmer');
const AsyncSender = require('../udp/async_sender');
const { PacketTypeEnum } = require("../../lib/udp");
const Transfer = require('../util/transfer');
const TraceHttpc = require('../trace/trace-httpc');

var GlobalObserver = function (agent) {
    this.agent = agent;
    this.packages = ['global'];
}

var transaction_status_error_enable = conf.getProperty('transaction_status_error_enable', true);
conf.on('transaction_status_error_enable', function (newProps) {
    transaction_status_error_enable = newProps;
})

GlobalObserver.prototype.inject = function (mod, moduleName) {
    var self = this;

    shimmer.wrap(mod, 'fetch', function(original) {
        return async function(...args) {
            var info = args[1] ? args[1] : {};
            var ctx = TraceContextManager._asyncLocalStorage.getStore();

            if (ctx) {
                ctx.start_time = Date.now(); // start_time 설정 추가

                TraceHelper.interTxTraceAutoOn(ctx);

                if (conf.getProperty('mtrace_enabled', false)) {
                    addTraceHeaders(info, ctx);
                    args[1] = info;
                }

                if (args[0]) {
                    const url = new URL(args[0]);
                    ctx.httpc_url = url.pathname + url.search;
                    ctx.httpc_host = url.hostname;
                    ctx.httpc_port = url.port || (url.protocol === 'https:' ? 443 : 80);
                    ctx.active_httpc_hash = true;
                }
            }

            try {
                const response = await original.apply(this, args);
                handleResponse(ctx, args, response);
                return response;
            } catch (e) {
                handleError(ctx, args, e);
                throw e;
            }
        };
    });
};

function addTraceHeaders(info, ctx) {
    // 헤더 객체 초기화
    if (!info.headers) {
        info.headers = {};
    }

    info.headers['x-wtap-po'] = Transfer.POID();

    if (conf.stat_mtrace_enabled) {
        info.headers[conf._trace_mtrace_spec_key1] = Transfer.SPEC_URL(ctx);
    }

    if (!ctx.mtid.isZero()) {
        info.headers[conf._trace_mtrace_caller_key] = Transfer.MTID_CALLERTX(ctx);
    }

    ctx.mcallee = KeyGen.getMtid();
    info.headers[conf._trace_mtrace_callee_key] = Hexa32.toString32(ctx.mcallee);
}

function handleResponse(ctx, args, response) {
    if (!ctx || !args[0]) return;

    // Error handling for 4xx, 5xx responses
    if (!response.ok && response.status >= 400 && transaction_status_error_enable) {
        let error = {
            class: response.statusText,
            message: ''
        }
        interceptorError(response.status, error, ctx);
    }

    endHttpc(ctx);
}

function handleError(ctx, args, err) {
    if (!ctx || !args[0]) return;

    const networkErrorToStatusCode = {
        'ECONNREFUSED': 503,
        'ETIMEDOUT': 504,
        'ENOTFOUND': 502,
        'ECONNRESET': 503,
        'EPIPE': 503,
        'EHOSTUNREACH': 503,
    };
    const statusCode = networkErrorToStatusCode[err.code] || 500;

    if (transaction_status_error_enable) {
        let error = {
            class: err.code || err.name,
            message: err.message || ''
        }
        interceptorError(statusCode, error, ctx);
    }

    endHttpc(ctx);
}

function interceptorError(statusCode, error, ctx) {
    if (!ctx) {
        return;
    }

    ctx.status = statusCode;
    let errors = [];
    let error_message = 'Request failed with status code ';

    if (statusCode >= 400 && !ctx.error) {
        ctx.error = 1;

        errors.push(error.class);
        if (error.message) {
            errors.push(error.message);
        } else {
            errors.push(error_message + statusCode);
        }

        AsyncSender.send_packet(PacketTypeEnum.TX_ERROR, ctx, errors);
    }
}

function endHttpc(ctx) {
    if (ctx == null) {
        return;
    }

    ctx.active_httpc_hash = false;
    let urls = `${ctx.httpc_host}:${ctx.httpc_port}${ctx.httpc_url}`;
    let httpcDatas = [urls, ctx.mcallee];
    ctx.elapsed = Date.now() - ctx.start_time;
    TraceHttpc.isSlowHttpc(ctx);
    AsyncSender.send_packet(PacketTypeEnum.TX_HTTPC, ctx, httpcDatas);
}

// interTxTraceAutoOn function moved to util/trace-helper.js

exports.GlobalObserver = GlobalObserver;