/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var platform = process.platform,
    arch     = process.arch,
    version  = parseFloat(process.version.slice(1));

var os      = require('os'),
    v8      = require('v8'),
    // gcstat  =  require('../../bindings/'+platform+'/'+arch+'/whatap.node'),
    Long    = require('long'),
    df      = require('df'),
    pusage  = require('pidusage');

function ResourceProfile() {
    var self = this;
    self.gc_count = 0;
    self.gc_time =0;
    self.cpu_info = {
        times   : 0,
        usage   : 0,
        core    : 0,
        proc    : 0
    };

    /** Disk usage */
    if(os.type() != 'Windows_NT'){
        setInterval(function () {
            df(function (err, table) {
                if(err || table == null) {
                    return self.disk = 0;
                }
                table.forEach(function(n, i){
                    if(n.mountpoint === '/'){
                        self.disk = n.percent;
                    }
                });
            });
        }, 1000);
    }

    /** Process cpu */
    setInterval(function () {
        pusage.stat(process.pid, function(err, stat) {
            if(err || stat == null) return;
            self.pcpu = stat.cpu;
            self.cpu_info.proc = stat.cpu;
        });
    }, 1000);

    /** Machine CPU */
    self.setCPUUsage();
    // gcstat.startMonitor(function (starttime, duration, type, flag) {
    //     // starttime => second
    //     if(self.gc_count == null) {
    //         self.gc_count = 0;
    //         self.gc_time = 0;
    //     }
    //     self.gc_count = self.gc_count.add(1);
    //     self.gc_time = self.gc_time.add(Number(duration));
    // });
}
ResourceProfile.prototype.setCPUUsage = function () {
    var self = this;
    var before = self.getCPUInfo();

    setTimeout(function () {

        var after = self.getCPUInfo();

        var idle = after.idle - before.idle;
        var total = after.total - before.total;

        var percent = idle / total * 100;
        self.cpu_info.times = total;
        self.cpu_info.usage = 100 - percent;
        self.cpu_info.core = os.cpus().length;

        setTimeout(function () {
            self.setCPUUsage();
        }, 100);

    }, 1000);

};
ResourceProfile.prototype.getCPUInfo = function () {
    var cpus = os.cpus();

    var user = 0;
    var nice = 0;
    var sys = 0;
    var idle = 0;
    var irq = 0;
    var total = 0;

    for(var cpu in cpus){
        if (!cpus.hasOwnProperty(cpu)) continue;
        user += cpus[cpu].times.user;
        nice += cpus[cpu].times.nice;
        sys += cpus[cpu].times.sys;
        irq += cpus[cpu].times.irq;
        idle += cpus[cpu].times.idle;
    }

    var total = user + nice + sys + idle + irq;

    return {
        'user' : user,
        'idle': idle,
        'total': total
    };
};
ResourceProfile.prototype.getCPU = function(){
    return this.cpu_info;
};
ResourceProfile.prototype.getMemory = function(){
    var stat = v8.getHeapStatistics();
    var data = {};
    data.heap_total = parseInt(stat.total_heap_size/1024);
    data.heap_used = parseInt(stat.used_heap_size/1024);
    data.usage = (os.totalmem() - os.freemem()) * 100/os.totalmem();

    return data;
};
ResourceProfile.prototype.getMemoryV2 = function(){
    var stat = v8.getHeapStatistics();
    var data = {};
    data.heap_total = parseInt(stat.total_heap_size);
    data.heap_used = parseInt(stat.used_heap_size);
    data.usage = (os.totalmem() - os.freemem()) * 100/os.totalmem();

    return data;
};
ResourceProfile.prototype.getDisk = function(){
    return {'usage' : this.disk};
};
ResourceProfile.prototype.getCPUTime = function () {
    var total = 0;
    var cpus = os.cpus();
    cpus.forEach(function(n){
        total += n.times['user'];
    });
    total = parseInt(total / cpus.length );
    return total;
};
ResourceProfile.prototype.getUsedHeapSize = function () {
    var stat = v8.getHeapStatistics();
    return stat.used_heap_size;
};
ResourceProfile.prototype.getCurrentProcGcInfo = function () {
    return [this.gc_count, this.gc_time];
};

module.exports = new ResourceProfile();