/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

const LOG2_BITS_PER_WORD    = 6;
const REGISTER_SIZE         = 5;

var Long = require('long');

function RegisterSet(count) {
    this.count = count;
    this.M = new Array(getSizeForCount(this.count));
    this.size = this.M.length;
}

RegisterSet.prototype.set = function (position, value) {
    var bucketPos = parseInt(position / LOG2_BITS_PER_WORD);
    var shift = REGISTER_SIZE * (position - (bucketPos * LOG2_BITS_PER_WORD));
    this.M[bucketPos] = (this.M[bucketPos] & ~(0x1f << shift) | (value << shift));
};
RegisterSet.prototype.get = function (position) {
    var bucketPos = parseInt(position / LOG2_BITS_PER_WORD);
    var shift = REGISTER_SIZE * (position - (bucketPos * LOG2_BITS_PER_WORD));
    return (this.M[bucketPos] & (0x1f << shift)) >>> shift;
};
RegisterSet.prototype.updateIfGreater = function (position, value) {
    var bucket = parseInt(position / LOG2_BITS_PER_WORD);
    var shift = REGISTER_SIZE * (position - (bucket * LOG2_BITS_PER_WORD));
    var mask = 0x1f << shift;

    var curVal = Long.fromNumber(this.M[bucket]).and(mask);
    var newVal = Long.fromNumber(value).shiftLeft(shift);

    if(newVal.greaterThan(curVal)) {
        var tmp = Long.fromNumber(this.M[bucket]).and(~mask).or(newVal);
        this.M[bucket] = tmp.getLowBits();
        return true;
    } else {
        return false;
    }
};
RegisterSet.prototype.merge = function (that) {

    for(var bucket = 0; bucket < this.M.length; bucket++) {
        var word = 0;
        for(var j=0; j<LOG2_BITS_PER_WORD; j++) {
            var mask = 0x1f << (REGISTER_SIZE * j);
            var thisVal = (this.M[bucket] & mask);
            var thatVal = (that.M[bucket] & mask);
            word |= (thisVal < thatVal) ? thatVal : thisVal;
        }
        this.M[bucket] = word;
    }
};
RegisterSet.prototype.readOnlyBits = function () {
    return this.M;
};

function getSizeForCount(count) {
    var bits = getBits(count);
    if(bits === 0) {
        return 1;
    } else if(bits % 32 === 0) {
        return bits;
    } else {
        return bits + 1;
    }
}
function getBits(count) {
    return parseInt(count / LOG2_BITS_PER_WORD);
}

module.exports = RegisterSet;