/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var RegisterSet = require('./registerset'),
    MurmurHash  = require('./murmurhash'),
    DataOutpuxX = require('../../io/data-outputx'),
    Long        = require('long');


function HyperLogLog() {
    this.log2m = 10;
    this.registerSet = new RegisterSet(1 << this.log2m);
    var m = 1 << this.log2m;
    this.alphaMM = getAlphaMM(this.log2m, m);
}

HyperLogLog.prototype.offerHashed = function (hashedValue) {
    var j = hashedValue >>> (32 - this.log2m);
    var r = numberOfLeadingZeros((hashedValue << this.log2m) | (1 << (this.log2m - 1)) + 1) + 1;
    return this.registerSet.updateIfGreater(j, r);
};
HyperLogLog.prototype.offer = function (o) {
    var val;
    if(o instanceof Long) {
        val = o;
    } else if(typeof o === 'string') {
        val = Long.fromString(o);
    } else if(typeof o === 'number') {
        val = Long.fromNumber(o);
    }
    var x = MurmurHash.hashLong(val);
    return this.offerHashed(x);
};
HyperLogLog.prototype.cardinality = function () {
    var registerSum = 0;
    var count = this.registerSet.count;
    var zeros = 0.0;
    for(var j=0; j<this.registerSet.count; j++) {
        var val = this.registerSet.get(j);
        registerSum += 1.0 / (1 << val);
        if(parseFloat(val) === 0.0) {
            zeros++;
        }
    }
    var estimate = this.alphaMM * (1 / registerSum);
    if(estimate <= (5.0 / 2.0) * count) {
        return Math.round(linearCounting(count, zeros));
    } else {
        return Math.round(estimate);
    }
};
HyperLogLog.prototype.getBytes = function () {
    var dout = new DataOutpuxX();
    dout.writeInt(this.log2m);
    dout.writeInt(this.registerSet.size);
    var M = this.registerSet.readOnlyBits();
    for(var i=0; i<M.length; i++) {
        dout.writeInt(M[i]);
    }
    return dout.toByteArray();
};

var getAlphaMM = function (p, m) {
    switch (p) {
        case 4:
            return 0.673 * m * m;
        case 5:
            return 0.697 * m * m;
        case 6:
            return 0.709 * m * m;
        default:
            return (0.7213 / (1 + 1.079 / m)) * m * m;
    }
};
var linearCounting = function (m, V) {
    return m * Math.log(m / V);
};
var numberOfLeadingZeros = function (i) {
    if (i === 0)
        return 32;
    var n = 1;

    if (i >>> 16 === 0) {
        n += 16;
        i <<= 16;
    }
    if (i >>> 24 === 0) {
        n += 8;
        i <<= 8;
    }
    if (i >>> 28 === 0) {
        n += 4;
        i <<= 4;
    }
    if (i >>> 30 === 0) {
        n += 2;
        i <<= 2;
    }
    n -= i >>> 31;
    return n;
}

module.exports = HyperLogLog;