/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var IntKeyMap       = require('../util/intkey-map'),
    HashUtil        = require('../util/hashutil'),
    TraceContext    = require('./trace-context'),
    Logger          = require('../logger'),
    EventLevel      = require('../data/event-level');
Long            = require('long');
var { AsyncLocalStorage } = require('async_hooks');
var crypto = require('crypto');

function TraceContextManager() {
    this.initialized = false;
    this.nextId = 1;
    this.currentId = null;
    this._asyncLocalStorage = new AsyncLocalStorage();
    this.entry = new IntKeyMap(1021,1).setMax(5000);
}

TraceContextManager.prototype.keys = function () {
    return this.entry.keys();
};
TraceContextManager.prototype.getContextEnumeration = function () {
    return this.entry.values();
};
TraceContextManager.prototype.size = function () {
    return this.entry.size();
};
TraceContextManager.prototype.getContext = function (key) {
    // if(key == null ) {
    //     return null;
    // }
    // return this.entry.get(key);
    return this._asyncLocalStorage.getStore();
};
TraceContextManager.prototype.getCurrentContext = function () {
    return this.getContext(this.currentId);
};
TraceContextManager.prototype.start = function () {

    if(this.initialized === false) { return null; }

    // var conf = require('../conf/configure');
    // if(this.size() > conf.profile_max_count) { return null; }

    var ctx = new TraceContext(this.getNextId());
    ctx.start_time = Date.now();
    this.entry.put(ctx._id, ctx);
    this.currentId = ctx._id;

    return ctx;
};
TraceContextManager.prototype.end = function (id) {
    if(id == null) {
        this.entry.remove(this.currentId); // important: high risky
        this.currentId = null;
    } else {
        this.entry.remove(id);
        // return;
    }
    // this._asyncLocalStorage.disable();
};
TraceContextManager.prototype.getNextId = function () {
    var length = 16
    var buffer = crypto.randomBytes(Math.ceil(length / 2));

    this.nextId =  buffer.toString('hex').substring(0, length);
    return this.nextId;
};
TraceContextManager.prototype.getCurrentId = function () {
    return this.currentId;
};
TraceContextManager.prototype.resume = function (id) {
    if(id == null) { return null; }
    if(id instanceof TraceContext) {
        id = id._id;
    }
    var c = this.getContext(id);
    if(c){
        this.currentId = id;
        return c;
    } else {
        this.currentId = null;
        return null;
    }
};
TraceContextManager.prototype.getTxProfile = function(n) {
    var ctx = this.getCurrentContext();
    if(ctx == null)
        return null;
    return ctx.profile.getLastSteps(n);
};
TraceContextManager.prototype.isExist = function (id) {
    return this.entry.containsKey(id);
};
var DataTextAgent = require('../data/datatext-agent');
var MessageStep = require('../step/message-step');
TraceContextManager.prototype.addStep = function (title, desc, ctx) {
    // var ctx = this.getCurrentContext();
    if(ctx == null) {
        return null;
    }
    try {
        var step = new MessageStep();
        step.start_time = ctx.getElapsedTime();
        step.hash = HashUtil.hashFromString(title);
        DataTextAgent.MESSAGE.add(step.hash, title);
        step.desc = desc;
        step.ctx = ctx;
        ctx.profile.push(step);
        return step;
    } catch(e) {
        return null;
    }
};
TraceContextManager.prototype.endStep = function (step) {
    if(step == null || step.ctx == null) {
        return;
    }
    this.resume(step.ctx);
    step.time = step.ctx.getElapsedTime() - step.start_time;
};

TraceContextManager.prototype.startTrace = function (title) {
    try{
        var ctx = this.start();
        if(ctx == null) {
            return null;
        }
        ctx.service_name = title;
        ctx.service_hash = HashUtil.hashFromString(title);
        var DataTextAgent = require('../data/datatext-agent'),
            ResourceProfile = require('../util/resourceprofile');
        DataTextAgent.SERVICE.add(ctx.service_hash, title);
        ctx.start_cpu = ResourceProfile.getCPUTime();
        ctx.start_malloc = ResourceProfile.getUsedHeapSize();

        return ctx;
    }catch(e){
        Logger.printError('WHATAP-998', 'TraceContext (startTrace)', e, true);
    }
};
TraceContextManager.prototype.endTrace = function (ctx) {
    if(ctx == null) {return;}
    var ProfilePack = require('../pack/profile-pack'),
        TxRecord  = require('../service/tx-record'),
        DateUtil    = require('../util/dateutil'),
        MeterService = require('../counter/meter/meter-service').MeterService,
        DataProfileAgent = require('../data/dataprofile-agent'),
        ResourceProfile = require('../util/resourceprofile'),
        SecurityMaster = require('../net/security-master');

    var profile = new ProfilePack();
    var service = new TxRecord();
    profile.time = Long.fromValue(ctx.start_time);
    service.endTime = DateUtil.currentTime();
    service.elapsed = ctx.getElapsedTime();
    service.service = ctx.service_hash;
    DataTextAgent.SERVICE.add(ctx.service_hash, ctx.service_name);

    service.cpuTime = ResourceProfile.getCPUTime() - ctx.start_cpu;
    service.malloc = ResourceProfile.getUsedHeapSize() - ctx.start_malloc;
    if(service.malloc < 0) { service.malloc = 0; }
    service.seq = ctx.txid;
    service.sqlCount = ctx.sql_count;
    service.sqlTime = ctx.sql_time;
    service.sqlFetchCount = ctx.rs_count;
    service.sqlFetchTime = parseInt(ctx.rs_time);
    service.ipaddr = ctx.remoteIp;
    service.userid = ctx.userid;

    service.mcaller_pcode = ctx.mcaller_pcode;
    service.mcaller_okind = ctx.mcaller_okind;
    service.mcaller_oid = ctx.mcaller_oid;

    if (ctx.error.isZero()==false/*long*/) {
        service.error = ctx.error;
        service.errorLevel = EventLevel.WARNING;
    }
    service.userAgent = ctx.userAgent;
    service.referer = ctx.referer;

    service.httpcCount = ctx.httpc_count;
    service.httpcTime = ctx.httpc_time;
    service.status = ctx.status;

    service.http_method=TxRecord.HTTP_METHOD[ctx.http_method] || TxRecord.WEB_GET;

    MeterService.add(service.service, service.elapsed, service.errorLevel, service.mcaller_pcode, service.mcaller_okind, service.mcaller_oid , service.apdex);

    profile.oid = SecurityMaster.OID;
    profile.service = service;

    var self = this;
    setTimeout(function () {
        DataProfileAgent.sendProfile(ctx, profile, false);
        self.end(ctx._id);
        ctx = null;
    }, 100);
};

module.exports = new TraceContextManager();