var { exec } = require('child_process');
var os = require('os');
var NodeInfo = require('./nodeinfo');
var Logger = require('../logger');

function StatusDetector() {
    this.process = async function () {
        let node = new NodeInfo();
        let stat = await this.netstat();

        if (!stat) {
            node = new NodeInfo();
        } else {
            try {
                node = this.parse(stat);
            } catch (e) {
                node = new NodeInfo();
                Logger.printError("WHATAP-203", "NodeInfo parse error", e, false);
            }
        }

        node.attr.type = 'nodejs';
        node.attr.time = Date.now();
        node.attr.ip = this.getLocalIpAddress();
        node.attr.pid = process.pid;
        node.attr.pname = process.title;

        return node;
    };

    this.netstat = function () {
        return new Promise((resolve, reject) => {
            let cmd = 'netstat -an -t';
            if (os.platform() === 'darwin') {
                cmd = 'netstat -an -p tcp';
            } else if (os.platform() === 'win32') {
                resolve('');
                return;
            }

            exec(cmd, (error, stdout, stderr) => {
                if (error) {
                    reject(error);
                    return;
                }
                resolve(stdout);
            });
        });
    };

    this.parse = function (netstat) {
        var node = new NodeInfo();
        var localIPs = this.getLocalIpSet();

        var lines = netstat.split('\n');
        lines.forEach(line => {
            if (line.startsWith('tcp')) {
                this.parseLine(node, line, localIPs);
            }
        });

        return node;
    };

    this.parseLine = function (node, line, localIPs) {
        var tokens = line.trim().split(/\s+/);
        if (!tokens[0].startsWith('tcp')) return;

        var localAddress = tokens[3].replace(/(\d+\.\d+\.\d+\.\d+)\.(\d+)/, '$1:$2');
        var remoteAddress = tokens[4].replace(/(\d+\.\d+\.\d+\.\d+)\.(\d+)/, '$1:$2');

        if (tokens[5] === 'LISTEN') {
            node.addListen(localIPs, localAddress);
        } else {
            node.addOutter(localAddress, remoteAddress);
        }
    };

    this.getLocalIpSet = function () {
        var ipSet = new Set();
        var interfaces = os.networkInterfaces();
        for (var devName in interfaces) {
            var iface = interfaces[devName];
            iface.forEach(alias => {
                if (alias.family === 'IPv4' && !alias.internal) {
                    ipSet.add(alias.address);
                }
            });
        }
        return ipSet;
    };

    this.getLocalIpAddress = function () {
        var interfaces = os.networkInterfaces();
        for (var devName in interfaces) {
            var iface = interfaces[devName];
            for (let i = 0; i < iface.length; i++) {
                var alias = iface[i];
                if (alias.family === 'IPv4' && !alias.internal) {
                    return alias.address;
                }
            }
        }
        return '0.0.0.0';
    };
}

StatusDetector.prototype = new StatusDetector();
StatusDetector.prototype.constructor = StatusDetector;

module.exports = StatusDetector;
