/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var LinkedMap       = require('../util/linkedmap'),
    StatHttpcPack   = require('../pack/stathttpc-pack'),
    HttpcRec        = require('../pack/stathttpc-rec'),
    DataPackSender  = require('../data/datapack-sender'),
    Logger          = require('../logger');

const TABLE_MAX_SIZE = 5000;

function HTTPC(url, host, port) {
    this.url = url || 0;
    this.host = host || 0;
    this.port = port || 0;
}

HTTPC.prototype.hashCode = function () {
    var prime = 31;
    var result = 1;
    result = prime * result + this.host;
    result = prime * result + this.port;
    result = prime * result + this.url;
    return result;
};
HTTPC.prototype.equals = function (other) {
    if(this === other) {
        return  true;
    }
    if(other == null) {
        return false;
    }
    return this.host === other.host && this.url === other.url && this.port === other.port;
};

function StatHttpc() {
    if(typeof StatHttpc.instance === 'object') {
        return StatHttpc.instance;
    }

    this.table = new LinkedMap(TABLE_MAX_SIZE + 1 , 1).setMax(TABLE_MAX_SIZE);
    this.table.create = function (key) {
        if(this.size() >= TABLE_MAX_SIZE) {
            return null;
        } 
        var rec = new HttpcRec();
        rec.url = key.url;
        rec.host = key.host;
        rec.port = key.port;
        rec.time_min = Number.MAX_SAFE_INTEGER;
        return rec;
    };
    StatHttpc.instance = this;
}

StatHttpc.prototype.getHttpc = function (url, host, port) {
    return this.table.intern(new HTTPC(url, host, port));
};
StatHttpc.prototype.addHttpcTime = function (service_hash, httpc_url_hash, httpc_host, httpc_port, time, isErr) {
    
    try {
        if(httpc_url_hash === 0) {
            return;
        }

        var httpcRec = this.getHttpc(httpc_url_hash, httpc_host, httpc_port);
        if(httpcRec != null) {
            httpcRec.count_total++;
            httpcRec.time_sum +=time;
            httpcRec.time_std +=time * time;
            httpcRec.time_max = Math.max(httpcRec.time_max, time);
            httpcRec.time_min = Math.min(httpcRec.time_min, time);
            if (isErr) {
                httpcRec.count_error++;
            }
            httpcRec.service = service_hash;
        }
    } catch(e) {
        Logger.printError("WHATAP-610", "StatHttpC addHttpcTime error", e, false);
    }
};
StatHttpc.prototype.send = function (now) {
    if(this.table.size() === 0) {
        return;
    }
    var p = new StatHttpcPack().setRecords(this.table.size(), this.table.values());
    p.time = now;
    this.table.clear();
    DataPackSender.sendStatHttpcPack(p);
};
StatHttpc.prototype.clear = function () {
    this.table.clear();
};

module.exports = new StatHttpc();