/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var fs              = require('fs'),
    path            = require('path'),
    MessageStep     = require('../step/message-step'),
    HashUtil        = require('../util/hashutil'),
    DataTextAgent   = require('../data/datatext-agent');

const emptyFunc = function () {};

function PlugIn(script) {
    this.script = script;
    this.lastModified = 0;
    this.process = emptyFunc;
    this.ctx = null;
}

PlugIn.prototype.create = function () {
    var self = this;
    var full_path = path.join(self.rootPath(), 'plugin', self.script);
    fs.stat(full_path, function (err, stats) {
        if(err) {
            self.process = emptyFunc();
            return;
        }
        var lastModifiedDate = stats.mtime;
        var inMillis = lastModifiedDate.getTime();

        if(self.lastModified < inMillis) {
            delete require.cache[require.resolve(full_path)];
            self.lastModified = inMillis;
            try{
                var func = require(full_path);
                if(typeof func === 'function') {
                    self.process = func;
                } else {
                    self.process = emptyFunc;
                }
            } catch(e) {
                self.process = emptyFunc;
            }
        }
    });
};
PlugIn.prototype._process = function () {
    var self = this;
    self.ctx = arguments[0];
    if(typeof self.process === 'function') {
        self.process.apply(this, arguments);
    }
};
PlugIn.prototype.rootPath = function () {
    return process.cwd();
};
PlugIn.prototype.print = function (str) {
    console.log(str)
};
PlugIn.prototype.profile = function (msg) {
    if(this.ctx == null) {
        return;
    }
    var hash = HashUtil.hashFromString(msg);
    DataTextAgent.MESSAGE.add(hash, msg);

    var step = new MessageStep();
    step.hash = hash;
    step.start_time = this.ctx.getElapsedTime();
    this.ctx.profile.add(step);
};

module.exports = PlugIn;
