const conf = require('../conf/configure'),
    StringKeyLinkedMap = require('../util/stringkey-linkedmap'),
    Logger = require('../logger');
const PiiItem = require('./pii-item');

class PiiMask {
    static instance = null;

    constructor() {
        // StringKeyLinkedMap을 직접 생성하고 create 메서드를 구현
        this.table = new StringKeyLinkedMap();
        this.table.create = function(key) {
            return [];
        };

        this.sqlMask = [];
        this.confLogSink = "";
        this.confSql = "";

        // IntLinkedSet 대신 Set 사용 (자바스크립트에서는 Set으로 충분)
        this.sql_p1_has = new Set();
        this.sql_p1_not = new Set();
        this.sql_p2_has = new Set();
        this.sql_p2_not = new Set();

        this.MAX_SET_SIZE = 5000;
    }

    static getInstance() {
        if (!PiiMask.instance) {
            PiiMask.instance = new PiiMask();
        }
        return PiiMask.instance;
    }

    static trimToEmpty(str) {
        return str ? str.trim() : '';
    }

    static isEmpty(str) {
        return !str || str.trim().length === 0;
    }

    static tokenizer(str, delimiter) {
        return str.split(delimiter).map(item => item.trim()).filter(item => item.length > 0);
    }

    static update() {
        PiiMask.getInstance().update();
    }

    static parseLogSink(pack) {
        return PiiMask.getInstance().parseLogSink(pack);
    }

    static parseSqlP1(sqlHash, param) {
        return PiiMask.getInstance().parseSqlP1(sqlHash, param);
    }

    static parseSqlP2(sqlHash, param) {
        return PiiMask.getInstance().parseSqlP2(sqlHash, param);
    }

    update() {
        this.updateLogSink();
        this.updateSql();
    }

    updateLogSink() {
        const val = PiiMask.trimToEmpty(conf.getProperty('pii_mask_logsink_items', ''));
        if (val === this.confLogSink) {
            return;
        }

        this.confLogSink = val;
        const tmp = new StringKeyLinkedMap();
        tmp.create = function(key) {
            return [];
        };

        const ids = PiiMask.tokenizer(val, ',');
        for (const id of ids) {
            const category = conf.getLogSinkProperty(id, 'category');
            const pattern = conf.getLogSinkProperty(id, 'pattern');
            const field = conf.getLogSinkProperty(id, 'field');

            if (PiiMask.isEmpty(category) || PiiMask.isEmpty(pattern)) {
                Logger.printError('WHATAP-901', `logsink pii config for [${id}] is empty or invalid`, false);
                continue;
            }

            var categories = PiiMask.tokenizer(category, ',');
            for(let _category of categories){
                tmp.intern(_category).push(new PiiItem(field, pattern));
            }
        }
        this.table = tmp;
    }

    updateSql() {
        const val = PiiMask.trimToEmpty(conf.getProperty('pii_mask_sql_items', ''));
        if (val === this.confSql) {
            return;
        }

        this.confSql = val;
        const tmp = [];

        const ids = PiiMask.tokenizer(val, ',');
        for (const id of ids) {
            const pattern = conf.getSqlProperty(id, 'pattern');
            if (PiiMask.isEmpty(pattern)) {
                Logger.printError('WHATAP-901', `sql pii config for [${id}] is empty or invalid`, false);
                continue;
            }
            tmp.push(new RegExp(pattern));
        }
        this.sqlMask = tmp;
    }

    parseLogSink(pack) {
        const list = this.table.get(pack.category);
        if (!list) {
            return;
        }

        for (const piiItem of list) {
            piiItem.mask(pack);
        }
    }

    parseSqlP1(sqlHash, param) {
        if (this.sqlMask.length === 0 || PiiMask.isEmpty(param)) {
            return param;
        }
        return this.exec(sqlHash, param, this.sql_p1_has, this.sql_p1_not);
    }

    parseSqlP2(sqlHash, param) {
        if (this.sqlMask.length === 0 || PiiMask.isEmpty(param)) {
            return param;
        }
        return this.exec(sqlHash, param, this.sql_p2_has, this.sql_p2_not);
    }

    exec(sqlHash, param, sqlHas, sqlNot) {
        // Maintain set size limit
        if (sqlHas.size > this.MAX_SET_SIZE) {
            sqlHas.clear();
        }
        if (sqlNot.size > this.MAX_SET_SIZE) {
            sqlNot.clear();
        }

        if (sqlNot.has(sqlHash)) {
            return param;
        }
        if (sqlHas.has(sqlHash)) {
            return conf.getProperty('pii_mask_sql_replace', 'pii-included');
        }

        for (const pattern of this.sqlMask) {
            if (pattern.test(param)) {
                sqlHas.add(sqlHash);
                return conf.getProperty('pii_mask_sql_replace', 'pii-included');
            }
        }

        sqlNot.add(sqlHash);
        return param;
    }
}

module.exports = PiiMask;