/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var Pack            = require('./pack'),
    PackEnum        = require('./packenum'),
    IntKeyLinkedMap = require('../util/intkey-linkedmap'),
    DataOutputX     = require('../io/data-outputx'),
    DataInputX      = require('../io/data-inputx'),
    IntKeyMap = require('../util/intkey-map'),
    TimeCount      = require('./time-count');

function TransactionRec() {
    this.hash = 0; /*int*/

    this.count = 0; /*int*/
    this.error = 0; /*int*/

    this.time_sum = 0; /*long*/
    this.time_max = 0; /*int*/
    
    this.sql_count = 0; /*int*/
    this.sql_time = 0; /*long*/
    this.sql_fetch = 0; /*int*/
    this.sql_fetch_time = 0; /*long*/

    this.httpc_count = 0; /*int*/
    this.httpc_time = 0; /*long*/
    this.malloc_sum = 0; /*long*/
    this.cpu_sum = 0; /*long*/

    this.sqlMap = null;     // int-key-map
    this.httpcMap = null;   // int-key-map

    /*임시변수 서버전송 안함*/  
    this.profiled = false;

    this.apdex_satisfied = 0; /*int*/
    this.apdex_tolerated = 0; /*int*/
    
    this.time_min = 0; /*int*/
    this.time_std = Long.ZERO; /*long*/
}

TransactionRec.prototype.mergeForStat = function(o){
    this.count += o.count;
    this.error += o.error;
    
    this.time_sum +=o.time_sum;
    this.time_max = Math.max(this.time_max, o.time_max);
    
    this.sql_count += o.sql_count;
    this.sql_time +=o.sql_time;
    this.sql_fetch += o.sql_fetch;
    this.sql_fetch_time +=o.sql_fetch_time;
    
    this.httpc_count += o.httpc_count;
    this.httpc_time +=o.httpc_time;
    
    this.malloc_sum +=o.malloc_sum;
    this.cpu_sum +=o.cpu_sum;
};

TransactionRec.prototype.merge = function(o) {
    this.mergeForStat(o);

    if(o.sqlMap !== undefined && o.sqlMap.size() > 0){
        if (this.sqlMap == undefined) {
            this.sqlMap = createMap(o.sqlMap.size());
            var en = o.sqlMap.entries();
            while (en.hasMoreElements()) {
                var e = en.nextElement();
                this.sqlMap.put(e.getKey(), e.getValue().copy());
            }
        }else{
            var en = o.sqlMap.entries();
            while (en.hasMoreElements()) {
                var e = en.nextElement();
                var tc = this.sqlMap.get(e.getKey());
                if (!tc) {
                    this.sqlMap.put(e.getKey(), e.getValue().copy());
                } else {
                    tc.merge(e.getValue());
                }
            }
        }
    }
};

TransactionRec.prototype.setUrlHash = function(hash) {
    this.hash = hash;
    return this;
};

TransactionRec.prototype.write = function(dout) {
        dout.writeInt(this.hash);
        dout.writeByte(2);
        dout.writeDecimal(this.count);
        dout.writeDecimal(this.error);
        dout.writeDecimal(this.time_sum);
        dout.writeDecimal(this.time_max);
        
        dout.writeDecimal(this.sql_count);
        dout.writeDecimal(this.sql_time);
        dout.writeDecimal(this.sql_fetch);
        dout.writeDecimal(this.sql_fetch_time);
        
        dout.writeDecimal(this.httpc_count);
        dout.writeDecimal(this.httpc_time);
        dout.writeDecimal(this.malloc_sum);
        dout.writeDecimal(this.cpu_sum);

        dout.writeDecimal(0);
        dout.writeDecimal(0);
};

TransactionRec.prototype.read = function(din) {
    this.hash = din.readInt();
    var ver = din.readByte();
    this.count = din.readDecNumber();
    this.error = din.readDecNumber();
    this.time_sum = din.readDecNumber();
    this.time_max = din.readDecNumber();

    this.sql_count = din.readDecNumber();
    this.sql_time = din.readDecNumber();
    this.sql_fetch = din.readDecNumber();
    this.sql_fetch_time = din.readDecNumber();
  
    this.httpc_count = din.readDecNumber();
    this.httpc_time = din.readDecNumber();
    this.malloc_sum = din.readDecNumber();
    this.cpu_sum = din.readDecNumber();

    _drop(din);
    _drop(din);
    /*
    var sqlcnt = din.readDecimal();
    if (sqlcnt > 0) {
        this.sqlMap = createMap(sqlcnt);
        for (var i = 0; i < sqlcnt; i++) {
            var hash = din.readInt();
            this.sqlMap.put(hash, new TimeCount().read(din));
        }
    }
    var httpcnt = din.readDecimal();
    if (httpcnt > 0) {
        this.httpcMap = createMap(httpcnt);
        for (var i = 0; i < httpcnt; i++) {
            var hash = din.readInt();
            this.httpcMap.put(hash,  new TimeCount().read(din));
        }
    }

    if(ver <=2) return this;
	this.apdex_satisfied=  din.readDecimal();
	this.apdex_tolerated=  din.readDecimal();

    if(ver <=3) return this;
	this.time_min= din.readDecimal();
	this.time_std= din.readDecimal();
    */
    return this;
};

TransactionRec.readRec = function(din){
    var urlHash = din.readInt();
    var ver = din.readByte();

    if(ver <= 1){
        var profiled = (ver==1);
        return;
    }

    var m = new TransactionRec(); 
    m.hash = urlHash;
    m.count = din.readDecimal();
    m.error = din.readDecimal();
    m.time_sum = din.readDecimal();
    m.time_max = din.readDecimal();

    m.sql_count = din.readDecimal();
    m.sql_time = din.readDecimal();
    m.sql_fetch = din.readDecimal();
    m.sql_fetch_time = din.readDecimal();

    m.httpc_count = din.readDecimal();
    m.httpc_time = din.readDecimal();
    m.malloc_sum = din.readDecimal();
    m.cpu_sum = din.readDecimal();

    var sqlcnt = din.readDecimal();
    if (sqlcnt > 0) {
        m.sqlMap = createMap(sqlcnt);
        for (var i = 0; i < sqlcnt; i++) {
            var hash = din.readInt();
            m.sqlMap.put(hash, new TimeCount().read(din));
        }
    }
    var httpcnt = din.readDecimal();
    if (httpcnt > 0) {
        m.httpcMap = createMap(httpcnt);
        for (var i = 0; i < httpcnt; i++) {
            var hash = din.readInt();
            m.httpcMap.put(hash,  new TimeCount().read(din));
        }
    }
    /*
    if(ver <=2) return m;
	m.apdex_satisfied=  din.readDecimal();
	m.apdex_tolerated=  din.readDecimal();

    if(ver <=3) return m;
	m.time_min= din.readDecimal();
	m.time_std= din.readDecimal();
    */
    return m;
}

TransactionRec.writeRec = function( o, m ) {
    let ver = 2;
    o.writeInt(m.hash);
    o.writeByte(ver); //0,1 = old version
    o.writeDecimal(m.count);
    o.writeDecimal(m.error);
    o.writeDecimal(m.time_sum);
    o.writeDecimal(m.time_max);
    o.writeDecimal(m.sql_count);
    o.writeDecimal(m.sql_time);
    o.writeDecimal(m.sql_fetch);
    o.writeDecimal(m.sql_fetch_time);

    o.writeDecimal(m.httpc_count);
    o.writeDecimal(m.httpc_time);
    o.writeDecimal(m.malloc_sum);
    o.writeDecimal(m.cpu_sum);

    if (m.sqlMap == null) {
        o.writeDecimal(0);
    } else {
        o.writeDecimal(m.sqlMap.size());
        var en = m.sqlMap.entries();        // IntKeyEntry
        while (en.hasMoreElements()) {
            var ent = en.nextElement();     // TimeCount
            o.writeInt(ent.getKey());
            ent.getValue().write(o);
        }
    }
    if (m.httpcMap == null) {
        o.writeDecimal(0);
    } else {
        o.writeDecimal(m.httpcMap.size());
        var en = m.httpcMap.entries();      // IntKeyEntry
        while (en.hasMoreElements()) {
            var ent = en.nextElement();     // TimeCount
            o.writeInt(ent.getKey());
            ent.getValue().write(o);
        }
    }
    /*
    if(ver <=2) return;
    o.writeDecimal(m.apdex_satisfied);
    o.writeDecimal(m.apdex_tolerated);		

    if(ver <=3) return;
    o.writeDecimal(m.time_min);
    o.writeDecimal(m.time_std);
    */
}

TransactionRec.createMap = function(cnt) {
    var m = new IntKeyMap(cnt);
    m.create = function(key){
        return this.size() >= 1000 ? null : new TimeCount();
    }
    return m;
}

function _drop(din){
    var c = din.readDecNumber();
     if (c > 0) {
         for (i = 0; i < c; i++) {
             din.readInt();
             din.readDecNumber();
             din.readDecNumber();
             din.readDecNumber();
         }
     }
}
module.exports = TransactionRec;
