/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var MeterSocketio = require('../counter/meter/meter-socket.io'),
    TraceContextManager = require('../trace/trace-context-manager'),
    SocketStep = require('../step/socket-step'),
    conf = require('../conf/configure'),
    IPUtil = require('../util/iputil'),
    Logger = require('../logger');
const {Detector: URLPatternDetector} = require("whatap/lib/trace/serviceurl-pattern-detector");
const HashUtil = require("whatap/lib/util/hashutil");
const DataTextAgent = require("whatap/lib/data/datatext-agent");
const ResourceProfile = require("whatap/lib/util/resourceprofile");
const ProfilePack         = require('../pack/profile-pack');
const TxRecord            = require('../service/tx-record');
const DateUtil            = require('../util/dateutil');
const SecurityMaster      = require('../net/security-master');
const DataProfileAgent    = require('../data/dataprofile-agent');
const MessageStep = require("whatap/lib/step/message-step");
const MeterUsers = require("whatap/lib/counter/meter/meter-users");
const DataPackSender = require("whatap/lib/data/datapack-sender");
const MeterService        = require('../counter/meter/meter-service').MeterService;

var trace_background_socket_enabled = conf.getProperty('trace_background_socket_enabled', true);
conf.on('trace_background_socket_enabled', function (newProps) {
    trace_background_socket_enabled = newProps;
})
var trace_sampling_enabled = conf.getProperty('trace_sampling_enabled', true);
conf.on('trace_sampling_enabled', function (newProps) {
    trace_sampling_enabled = newProps;
})
var trace_sampling_tps = conf.getProperty('trace_sampling_tps', 1000);
conf.on('trace_sampling_tps', function (newProps) {
    trace_sampling_tps = newProps;
})

var WebsocketObserver = function(agent){
    this.agent = agent;
    this.packages = ['websocket'];
};

var socket_count = {
    count: 0,
    start_time: null
};

WebsocketObserver.prototype.inject = function (mod, moduleName) {
    if (mod.__whatap_observe__) {
        return;
    }
    mod.__whatap_observe__ = true;
    Logger.initPrint("WebsocketObserver");

    var self = this;
    var aop = self.agent.aop;

    aop.after(mod.server.prototype, 'on', function (obj, args, ret, lctx){
        aop.after(mod.connection.prototype, ['send', 'sendUTF', 'sendBytes'], function (obj, args, ret, lctx) {
            if(!trace_background_socket_enabled) {
                return;
            }
            if(trace_sampling_enabled){
                var now = Date.now();
                if(!socket_count.start_time) {
                    socket_count.start_time = now;
                }

                if ((now - socket_count.start_time) > 5000) {
                    socket_count.start_time = now;
                    socket_count.count = 0;
                }

                socket_count.count++;
                if(socket_count.count > trace_sampling_tps) {
                    MeterService.add(0, 1, 0, 0, 0, 0);
                    return;
                }
            }

            var protocol = obj.protocol;
            var remoteAddress = obj.remoteAddress;

            TraceContextManager._asyncLocalStorage.run(initCtx(obj, args), () => {
                try{
                    var ctx = TraceContextManager._asyncLocalStorage.getStore();
                    if (!ctx) {
                        return;
                    }

                    var host;
                    if (remoteAddress && remoteAddress.includes(':')) {
                        host = remoteAddress.substring(remoteAddress.lastIndexOf(':') + 1);
                    }

                    ctx.socket_connecting = true;
                    ctx.footprint('Websocket Connecting: ' + host);

                    if (protocol) {
                        var protocol_hash = HashUtil.hashFromString('Protocol');
                        var step = new MessageStep();
                        step.hash = protocol_hash;
                        step.start_time = ctx.getElapsedTime();
                        step.desc = protocol;

                        DataTextAgent.MESSAGE.add(protocol_hash, 'Protocol');
                        ctx.profile.add(step);
                    }

                    var step = new SocketStep();
                    step.start_time = ctx.getElapsedTime();
                    step.ipaddr = Buffer.from(IPUtil.stringToBytes(host));
                    // step.port = port;

                    ctx.socket_connecting = false;
                    step.elapsed = ctx.getElapsedTime() - step.start_time;
                    ctx.profile.push(step);

                    ctx.footprint('Websocket Connecting Done');

                    self.__endTransaction(null, ctx);
                    return;
                }catch (e) {
                    Logger.printError('WHATAP-616', 'Websocket transaction error..', e, false);
                    return;
                }
            });
        })
    })
};

WebsocketObserver.prototype.__endTransaction = function(error, ctx) {
    try {
        var profile = new ProfilePack();
        var wtx = new TxRecord();
        wtx.endTime = DateUtil.currentTime();
        profile.time  = wtx.endTime;
        wtx.elapsed = ctx.getElapsedTime();

        DataTextAgent.SERVICE.add(ctx.service_hash, ctx.service_name);

        wtx.seq = ctx.txid;
        wtx.service = ctx.service_hash;
        wtx.cpuTime = ResourceProfile.getCPUTime() - ctx.start_cpu;
        wtx.malloc = ResourceProfile.getUsedHeapSize()-ctx.start_malloc;
        if(wtx.malloc < 0) { wtx.malloc = 0; }
        wtx.status = 2;

        wtx.ipaddr = ctx.remoteIp;

        MeterService.add(wtx.service, wtx.elapsed,
            wtx.errorLevel, ctx.mcaller_pcode, ctx.mcaller_okind, ctx.mcaller_oid);

        profile.oid = SecurityMaster.OID;
        profile.service = wtx;

        TraceContextManager.end(ctx._id);

        setTimeout(function () {
            DataProfileAgent.sendProfile(ctx, profile, false);
            TraceContextManager.end(ctx._id);
            ctx = null;
        }, 100);
    } catch (e) {
        Logger.printError('WHATAP-615', 'Websocket end transaction error..', e, false);
        TraceContextManager.end(ctx._id);
        ctx = null;
    }

};

function initCtx(socket, args) {
    const ctx = TraceContextManager.start();
    if (!ctx) {return;}

    var remote_addr;
    const address = socket.remoteAddress;
    if(address && address.includes(':')){
        remote_addr = address.substring(address.lastIndexOf(':')+1);
    }

    ctx.start_malloc = ResourceProfile.getUsedHeapSize();
    ctx.start_cpu = ResourceProfile.getCPUTime();

    remote_addr=IPUtil.checkIp4(remote_addr);
    ctx.remoteIp = IPUtil.stringToInt(remote_addr);
    ctx.userid = Long.fromNumber(ctx.remoteIp);
    MeterUsers.add(ctx.userid);

    return ctx;
}

exports.WebsocketObserver = WebsocketObserver;