/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var TraceContextManager = require('../trace/trace-context-manager'),
    SocketStep = require('../step/socket-step'),
    conf = require('../conf/configure'),
    IPUtil = require('../util/iputil'),
    Logger = require('../logger');
const {Detector: URLPatternDetector} = require("../trace/serviceurl-pattern-detector");
const DataTextAgent = require("../data/datatext-agent");
const ResourceProfile = require("../util/resourceprofile");
const ProfilePack         = require('../pack/profile-pack');
const TxRecord            = require('../service/tx-record');
const DateUtil            = require('../util/dateutil');
const SecurityMaster      = require('../net/security-master');
const DataProfileAgent    = require('../data/dataprofile-agent');
const MeterUsers = require("../counter/meter/meter-users");
const MeterService        = require('../counter/meter/meter-service').MeterService;
const shimmer = require('../core/shimmer');

var trace_background_socket_enabled = conf.getProperty('trace_background_socket_enabled', true);
conf.on('trace_background_socket_enabled', function (newProps) {
    trace_background_socket_enabled = newProps;
})
var trace_sampling_enabled = conf.getProperty('trace_sampling_enabled', true);
conf.on('trace_sampling_enabled', function (newProps) {
    trace_sampling_enabled = newProps;
})
var trace_sampling_tps = conf.getProperty('trace_sampling_tps', 1000);
conf.on('trace_sampling_tps', function (newProps) {
    trace_sampling_tps = newProps;
})

var SocketIOObserver = function(agent){
    this.agent = agent;
    this.packages = ['socket.io'];
};

var socket_count = {
    count: 0,
    start_time: null
};

SocketIOObserver.prototype.inject = function (mod, moduleName) {
    if (mod.__whatap_observe__) {
        return;
    }
    mod.__whatap_observe__ = true;
    Logger.initPrint("SocketIOObserver");

    var self = this;

    shimmer.wrap(mod.prototype, 'on', function (original) {
        return function (event, listener) {
            if (event === 'connection') {
                return original.call(this, event, function (socket) {
                    shimmer.wrap(socket, 'emit', function (origEmit) {
                        return function (emitEvent, ...args) {
                            if (trace_background_socket_enabled) {
                                self.__handleSocketEmitEvent(socket, emitEvent, args);
                            }

                            return origEmit.apply(this, [emitEvent, ...args]);
                        };
                    });

                    return listener.apply(this, [socket]);
                });
            }

            return original.call(this, event, listener);
        };
    });
};

SocketIOObserver.prototype.__handleSocketEmitEvent = function(socket, event, args) {
    if (trace_sampling_enabled) {
        var now = Date.now();
        // if ((now - socket_count.start_time) > 1000) {
        //     socket_count.start_time = now;
        //     socket_count.count = 0;
        // }
        if (!socket_count.start_time || (now - socket_count.start_time) >= 1000) {
            socket_count.start_time = now;
            socket_count.count = 0;
        }

        socket_count.count++;
        if (socket_count.count > trace_sampling_tps) {
            MeterService.add(0, 1, 0, SecurityMaster.PCODE, SecurityMaster.OKIND, SecurityMaster.OID);
            return;
        }
    }

    TraceContextManager._asyncLocalStorage.run(initCtx(socket, args), () => {
        try {
            var ctx = TraceContextManager._asyncLocalStorage.getStore();
            if (!ctx) return;

            ctx.footprint('Socket Emit Event: ' + event);

            var host;
            if (socket.conn && socket.conn.remoteAddress && socket.conn.remoteAddress.includes(':')) {
                host = socket.conn.remoteAddress.substring(socket.conn.remoteAddress.lastIndexOf(':') + 1);
            }

            ctx.socket_connecting = true;
            ctx.footprint('Socket Connecting: ' + host);

            var step = new SocketStep();
            step.start_time = ctx.getElapsedTime();
            step.ipaddr = Buffer.from(IPUtil.stringToBytes(host));
            // step.port = port;

            ctx.socket_connecting = false;
            step.elapsed = ctx.getElapsedTime() - step.start_time;
            ctx.profile.push(step)

            this.__endTransaction(null, ctx);

        } catch (e) {
            Logger.printError('WHATAP-616', 'socket.io emit transaction error..', e, false);
        }
    });
};

SocketIOObserver.prototype.__endTransaction = function(error, ctx) {
    try {
        var profile = new ProfilePack();
        var wtx = new TxRecord();
        wtx.endTime = DateUtil.currentTime();
        profile.time  = wtx.endTime;
        wtx.elapsed = ctx.getElapsedTime();

        DataTextAgent.SERVICE.add(ctx.service_hash, ctx.service_name);

        wtx.seq = ctx.txid;
        wtx.service = ctx.service_hash;
        wtx.cpuTime = ResourceProfile.getCPUTime() - ctx.start_cpu;
        wtx.malloc = ResourceProfile.getUsedHeapSize()-ctx.start_malloc;
        if(wtx.malloc < 0) { wtx.malloc = 0; }
        wtx.status = 2;

        wtx.ipaddr = ctx.remoteIp;

        MeterService.add(0, wtx.elapsed, 0, SecurityMaster.PCODE, SecurityMaster.OKIND, SecurityMaster.OID);

        profile.oid = SecurityMaster.OID;
        profile.service = wtx;

        setTimeout(function () {
            DataProfileAgent.sendProfile(ctx, profile, false);
            TraceContextManager.end(ctx._id);
            ctx = null;
        }, 100);
    } catch (e) {
        Logger.printError('WHATAP-615', 'Socket.io end transaction error..', e, false);
        TraceContextManager.end(ctx._id);
        ctx = null;
    }

};

function initCtx(socket, args) {
    const ctx = TraceContextManager.start();
    if (!ctx) {return;}

    var remote_addr;
    const address = socket.conn.remoteAddress;
    if(address && address.includes(':')){
        remote_addr = address.substring(address.lastIndexOf(':')+1);
    }

    ctx.start_malloc = ResourceProfile.getUsedHeapSize();
    ctx.start_cpu = ResourceProfile.getCPUTime();

    remote_addr=IPUtil.checkIp4(remote_addr);
    ctx.remoteIp = IPUtil.stringToInt(remote_addr);
    ctx.userid = Long.fromNumber(ctx.remoteIp);
    MeterUsers.add(ctx.userid);

    return ctx;
}

exports.SocketIOObserver = SocketIOObserver;