/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

const TraceContextManager = require('../trace/trace-context-manager');
const conf = require('../conf/configure');
const LogTracer = require('../logsink/log-tracer');

let logsink_enabled = conf.getProperty('logsink_enabled', false);
let logTracer = logsink_enabled ? new LogTracer() : null;

conf.on('logsink_enabled', function(newProperty) {
    logsink_enabled = newProperty;
    logTracer = logsink_enabled ? new LogTracer() : null;
});

const ProcessObserver = function (agent) {
    this.agent = agent;
    this.packages = ['process'];
};

ProcessObserver.prototype.inject = function (mod, moduleName) {
    this._hookNextTick(mod);
    this._hookStdOutWrite();
    this._hookStdErrWrite();
};

ProcessObserver.prototype._hookNextTick = function (mod) {
    this.agent.aop.before(mod, 'nextTick', (obj, args) => {
        const cached_id = TraceContextManager.getCurrentId();
        this.agent.aop.functionHook(args, -1, (obj, args) => {
            if (cached_id != null) {
                TraceContextManager.resume(cached_id);
            }
        });
    });
};

ProcessObserver.prototype._hookStdOutWrite = function () {
    this.agent.aop.after(process.stdout, 'write', (obj, args) => {
        if (conf.getProperty('logsink_enabled', false) && args[0]) {
            let content = typeof args[0] === 'string' ? args[0] : (args[0].message || '');
            try {
                const parsedContent = JSON.parse(content);
                content = parsedContent.message ? parsedContent.message : JSON.stringify(parsedContent);
            } catch (e) {
            }
            if (logTracer && content) {
                logTracer.addStdWrite(content, conf.logsink_category_stdout);
            }
        }
    });
};

ProcessObserver.prototype._hookStdErrWrite = function () {
    this.agent.aop.after(process.stderr, 'write', (obj, args) => {
        if (conf.getProperty('logsink_enabled', false) && args[0]) {
            let content = typeof args[0] === 'string' ? args[0] : (args[0].message || '');
            try {
                const parsedContent = JSON.parse(content);
                content = parsedContent.message ? parsedContent.message : JSON.stringify(parsedContent);
            } catch (e) {
            }
            if (logTracer && content) {
                logTracer.addStdWrite(content, conf.logsink_category_stderr);
            }
        }
    });
};

module.exports.ProcessObserver = ProcessObserver;