/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var TraceContextManager = require('../trace/trace-context-manager'),
    ParsedSql           = require('../trace/parsed-sql'),
    SqlStepX            = require('../step/sql-stepx'),
    DBCStep             = require('../step/dbc-step'),
    DataTextAgent       = require('../data/datatext-agent'),
    StatSql             = require('../stat/stat-sql'),
    MeterSql            = require('../counter/meter/meter-sql'),
    conf                = require('../conf/configure'),
    IntKeyMap           = require('../util/intkey-map'),
    EscapeLiteralSQL    = require('../util/escape-literal-sql'),
    HashUtil            = require('../util/hashutil'),
    StatError           = require('../stat/stat-error'),
    TextTypes           = require('../lang/text-types'),
    ParamSecurity       = require('../util/paramsecurity'),
    Logger              = require('../logger'),
    Buffer              = require('buffer').Buffer,
    DateUtil            = require('../util/dateutil'),
    TraceSQL                 = require('../trace/trace-sql');
const shimmer = require('../core/shimmer');
const ResultSetStep = require("../step/resultset-step");

var PgSqlObserver = function (agent) {
    this.agent = agent;
    this.packages = ['pg'];
};

var dbc_hash = 0;
var dbc, dbc_step;
PgSqlObserver.prototype.inject = function (mod, moduleName) {
    if (mod.__whatap_observe__) { return; }
    mod.__whatap_observe__ = true;

    Logger.initPrint("PgSqlObserver");

    var self = this;
    var aop = self.agent.aop;

    if (conf.sql_enabled === false) { return; }

    shimmer.wrap(mod.Client.prototype, 'connect', function (original) {
        return function () {
            const ctx = TraceContextManager.getCurrentContext();
            if (!ctx || ctx.db_opening) {
                return original.apply(this, arguments);
            }

            ctx.pg_opening = true;
            ctx.footprint('PgSql Connecting Start');

            dbc_step = new DBCStep();
            dbc_step.start_time = ctx.getElapsedTime();

            const finishConnect = function (param) {
                if (ctx) {
                    if (dbc_hash === 0) {
                        const info = param.connectionParameters;
                        dbc = 'postgresql://';
                        dbc += (info.user || '') + '@';
                        dbc += (info.host || '') + '/';
                        dbc += info.database || '';

                        dbc_hash = HashUtil.hashFromString(dbc);

                        // DataTextAgent.DBC.add(dbc_hash, dbc);
                        // DataTextAgent.METHOD.add(dbc_hash, dbc);
                        // DataTextAgent.ERROR.add(dbc_hash, dbc);
                    }
                    //
                    // dbc_step.hash = dbc_hash;
                    // dbc_step.elapsed = ctx.getElapsedTime() - dbc_step.start_time;
                    //
                    // ctx.db_opening = false;
                    // ctx.footprint('PgSql Connecting Done');
                    // ctx.profile.push(dbc_step);
                }
            };

            try {
                const result = original.apply(this, arguments);

                if (result && result.then) {
                    result.then(finishConnect(this));
                } else {
                    finishConnect(this);
                }

            } catch (err) {
                ctx.db_opening = false;
                Logger.printError('PgSqlObserver', 'Connect Error', err);
                throw err;
            }
        };
    });

    shimmer.wrap(mod.Client.prototype, 'query', function (original) {
        return function () {
            const ctx = TraceContextManager.getCurrentContext();
            if (!ctx) {
                return original.apply(this, arguments);
            }

            if(dbc && dbc_hash){
                if(!dbc_step){
                    dbc_step = new DBCStep();
                    dbc_step.start_time = ctx.getElapsedTime();
                }
                dbc_step.elapsed = ctx.getElapsedTime() - dbc_step.start_time;
                dbc_step.hash = dbc_hash;

                DataTextAgent.DBC.add(dbc_hash, dbc);
                DataTextAgent.METHOD.add(dbc_hash, dbc);
                DataTextAgent.ERROR.add(dbc_hash, dbc);

                ctx.pg_opening = false;
                ctx.footprint('PgSql Connecting Done');
                ctx.profile.push(dbc_step);
            }
            dbc_step = null;

            var sql_step = new SqlStepX();
            sql_step.start_time = ctx.getElapsedTime();
            ctx.profile.push(sql_step);
            ctx.footprint('PgSql Query Start');

            ctx.vvv++;

            let sql = arguments[0];
            let psql = null;
            if (typeof sql === 'string' && sql.length > 0) {
                try {
                    psql = escapeLiteral(sql);
                } catch (e) {
                    Logger.printError('WHATAP-191', 'PgSqlObserver escapeliteral error', e);
                }
            } else {
                sql = '';
                psql = escapeLiteral(sql);
            }

            if(psql != null) {
                sql_step.hash = psql.sql;
            }
            sql_step.dbc = dbc_hash;

            var els = new EscapeLiteralSQL(sql);
            els.process();
            ctx.active_sqlhash = sql_step.hash;
            ctx.active_dbc = sql_step.dbc;
            //ctx.active_crud = sql_step.crud;

            if(conf.profile_sql_param_enabled) {
                var params = arguments.length > 1 && Array.isArray(arguments[1]) ? arguments[1] : undefined;
                sql_step.setTrue(1);
                var crc = {value : 0};
                sql_step.p1 = toParamBytes(psql.param, crc);
                if(params != undefined) {
                    const result = params.map((param) => {
                        if(typeof param === 'string'){
                            return `'${param}'`
                        }
                        return param
                    }).toString()
                    sql_step.p2 = toParamBytes(result, crc);
                }
                sql_step.pcrc = crc.value;
            }

            try {
                const result = original.apply(this, arguments);

                if (result && result.then) {
                    return result.then(res => {
                        if(res.command && res.command === 'SELECT'){
                            var result_step = new ResultSetStep();
                            result_step.start_time = ctx.getElapsedTime();
                            result_step.elapsed = 0;
                            result_step.fetch = res.rowCount;
                            result_step.sqlhash = psql.sql;
                            result_step.dbc = dbc_hash;
                            ctx.profile.push(result_step);

                            ctx.rs_count = ctx.rs_count ? ctx.rs_count + res.rowCount : res.rowCount;

                            MeterSql.addFetch(result_step.dbc, result_step.fetch, 0);
                            StatSql.addFetch(result_step.dbc, result_step.sqlhash, result_step.fetch, 0);

                            TraceSQL.isTooManyRecords(sql_step, result_step.fetch, ctx);
                        }
                        self._finishQuery(ctx, sql_step);
                        return res;
                    }).catch(err => {
                        self._handleError(ctx, sql_step, err)
                        if (conf.trace_sql_error_stack && conf.trace_sql_error_depth) {
                            var traceDepth = conf.trace_sql_error_depth;

                            var errorStack = err.stack.split("\n");
                            if (errorStack.length > traceDepth) {
                                errorStack = errorStack.slice(0, traceDepth + 1);
                            }
                            ctx.error_message = errorStack.join("\n");
                            sql_step.error = ctx.error = StatError.addError('pgsql -' + err.code, err.message, ctx.service_hash, TextTypes.SQL, null);
                        }
                        throw err;
                    });
                } else {
                    const callback = arguments[arguments.length - 1];
                    if (typeof callback === 'function') {
                        arguments[arguments.length - 1] = function (err, res) {
                            if (err) {
                                self._handleError(ctx, sql_step, err);
                            } else {
                                self._finishQuery(ctx, sql_step);
                            }
                            return callback.apply(this, arguments);
                        };
                    }
                    return result;
                }
            } catch (err) {
                self._handleError(ctx, sql_step, err);
                throw err;
            }
        };
    });
};

PgSqlObserver.prototype._finishQuery = function (ctx, sql_step) {
    sql_step.elapsed = ctx.getElapsedTime() - sql_step.start_time;
    TraceSQL.isSlowSQL(ctx);

    MeterSql.add(sql_step.hash, sql_step.elapsed, false);
    StatSql.addSqlTime(ctx, ctx.service_hash, sql_step.dbc, sql_step.hash, sql_step.elapsed, false, 0);
};

PgSqlObserver.prototype._handleError = function (ctx, sql_step, err) {
    sql_step.elapsed = ctx.getElapsedTime() - sql_step.start_time;
    TraceSQL.isSlowSQL(ctx);

    MeterSql.add(sql_step.hash, sql_step.elapsed, false);
    StatSql.addSqlTime(ctx, ctx.service_hash, sql_step.dbc, sql_step.hash, sql_step.elapsed, true, 0);
};

var toParamBytes = function(p, crc) {
    if(p == null || p.length === 0) {
        return null;
    }
    try{
        return ParamSecurity.encrypt(Buffer.from(p), crc);
    } catch(e) {
        return null;
    }
};

var checkedSql = new IntKeyMap(2000).setMax(2000);
var nonLiteSql = new IntKeyMap(5000).setMax(5000);
var date = DateUtil.yyyymmdd();

function escapeLiteral(sql) {
    if(sql == null) { sql = ''; }

    if(date !== DateUtil.yyyymmdd()) {
        checkedSql.clear();
        nonLiteSql.clear();
        date = DateUtil.yyyymmdd();
        Logger.print('WHATAP-SQL-CLEAR', 'PgSqlObserver CLEAR OK!!!!!!!!!!!!!!!!', false);
    }

    var sqlHash = HashUtil.hashFromString(sql);
    var psql = nonLiteSql.get(sqlHash);

    if(psql != null) {
        return psql;
    }
    psql = checkedSql.get(sqlHash);

    if(psql != null) {
        return psql;
    }

    var els = new EscapeLiteralSQL(sql);
    els.process();

    var hash = HashUtil.hashFromString(els.getParsedSql());
    DataTextAgent.SQL.add(hash, els.getParsedSql());

    if(hash === sqlHash) {
        psql = new ParsedSql(els.sqlType, hash, null);
        nonLiteSql.put(sqlHash, psql);
    } else {
        psql = new ParsedSql(els.sqlType, hash, els.getParameter());
        checkedSql.put(sqlHash, psql);
    }
    return psql;
}

exports.PgSqlObserver = PgSqlObserver;