/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var TraceContextManager = require('../trace/trace-context-manager'),
    ParsedSql = require('../trace/parsed-sql'),
    SqlStepX = require('../step/sql-stepx'),
    DBCStep = require('../step/dbc-step'),
    ResultSetStep = require('../step/resultset-step'),
    DataTextAgent = require('../data/datatext-agent'),
    StatSql = require('../stat/stat-sql'),
    MeterSql = require('../counter/meter/meter-sql'),
    conf = require('../conf/configure'),
    IntKeyMap = require('../util/intkey-map'),
    EscapeLiteralSQL = require('../util/escape-literal-sql'),
    HashUtil = require('../util/hashutil'),
    StatError = require('../stat/stat-error'),
    TextTypes = require('../lang/text-types'),
    ParamSecurity = require('../util/paramsecurity'),
    Logger = require('../logger'),
    conf = require('../conf/configure'),
    DateUtil = require('../util/dateutil'),
    Buffer = require('buffer').Buffer,
    TraceSQL = require('../trace/trace-sql');

var MysqlObserver = function (agent) {
    this.agent = agent;
    this.packages = ['mysql', 'mysql2'];
};

var queryHook = function (dbc, agent) {

    return function (obj, args) {
        var ctx = TraceContextManager.getCurrentContext();

        if (ctx == null || args[0] == null) {
            return;
        }
        if (args[0].sql == null && typeof args[0] != 'string') {
            return;
        }

        var dbc_step = new DBCStep();
        var dbc_hash = HashUtil.hashFromString(dbc);
        DataTextAgent.DBC.add(dbc_hash, dbc);
        DataTextAgent.METHOD.add(dbc_hash, dbc);
        DataTextAgent.ERROR.add(dbc_hash, dbc)

        dbc_step.hash = dbc_hash;
        dbc_step.start_time = ctx.getElapsedTime();
        ctx.profile.push(dbc_step);

        var sql_step = new SqlStepX();
        sql_step.start_time = ctx.getElapsedTime();
        ctx.profile.push(sql_step);

        ctx.footprint('MySql Query Start');

        ctx.sql_count++;

        var sql = args.length > 0 ? args[0] : undefined,
            psql = null;

        if (typeof sql !== 'string') {
            sql = args[0].sql || undefined;
        }

        if (typeof sql === 'string' && sql.length > 0) {
            try {
                psql = escapeLiteral(sql);
            } catch (e) {
                Logger.printError('WHATAP-191', 'MysqlObserver escapeliteral error', e);
            }
        } else {
            sql = '';
            psql = escapeLiteral(sql);
        }

        if (psql != null) {
            sql_step.hash = psql.sql;
            // sql_step.crud = psql.type.charCodeAt(0);
        }
        sql_step.dbc = dbc_hash;

        var els = new EscapeLiteralSQL(sql);
        els.process();

        ctx.active_sqlhash = sql_step.hash;
        ctx.active_dbc = sql_step.dbc;
        //ctx.active_crud = sql_step.crud;

        if (conf.profile_sql_param_enabled) {
            var params = args.length > 1 && Array.isArray(args[1]) ? args[1] : undefined;
            sql_step.setTrue(1);
            var crc = {value: 0};
            sql_step.p1 = toParamBytes(psql.param, crc);
            if (params != undefined) {
                const result = params.map((param) => {
                    if (typeof param === 'string') {
                        return `'${param}'`
                    }
                    return param
                }).toString()
                sql_step.p2 = toParamBytes(result, crc);
            }
            sql_step.pcrc = crc.value;
        }

        var lastCallback = false;

        function queryCallback(obj, args) {
            if (ctx == null) {
                return;
            }
            TraceContextManager.resume(ctx._id);

            if (args[0]) {
                try {
                    if (conf.trace_sql_error_stack && conf.trace_sql_error_depth) {
                        var traceDepth = conf.trace_sql_error_depth;

                        var errorStack = args[0].stack.split("\n");
                        if (errorStack.length > traceDepth) {
                            errorStack = errorStack.slice(0, traceDepth + 1);
                        }
                        ctx.error_message = errorStack.join("\n");
                        sql_step.error = ctx.error = StatError.addError('mysql-' + args[0].code, args[0].sqlMessage, ctx.service_hash, TextTypes.SQL, null);
                    }
                    if (conf._is_trace_ignore_err_cls_contains === true && args[0].code.indexOf(conf.trace_ignore_err_cls_contains) < 0) {
                        sql_step.error = StatError.addError('mysql-' + args[0].code, args[0].message || 'mysql error', ctx.service_hash, TextTypes.SQL, sql_step.hash); /*long*/
                        if (ctx.error.isZero()) {
                            ctx.error = sql_step.error;
                        }
                    } else if (conf._is_trace_ignore_err_msg_contains === true && args[0].message.indexOf(conf.trace_ignore_err_msg_contains) < 0) {
                        sql_step.error = StatError.addError('mysql-' + args[0].code, args[0].message || 'mysql error', ctx.service_hash, TextTypes.SQL, sql_step.hash); /*long*/
                        if (ctx.error.isZero()) {
                            ctx.error = sql_step.error;
                        }
                    } else if (conf._is_trace_ignore_err_cls_contains === false && conf._is_trace_ignore_err_msg_contains === false) {
                        sql_step.error = StatError.addError('mysql-' + args[0].code, args[0].message || 'mysql error', ctx.service_hash, TextTypes.SQL, sql_step.hash); /*long*/
                        if (ctx.error.isZero()) {
                            ctx.error = sql_step.error;
                        }
                    }
                    /*
                     sql_step.error = StatError.addError(
                     ('mysql-'+args[0].code ),
                     (args[0].message || 'mysql error'),
                      ctx.service_hash,
                      TextTypes.SQL, sql_step.hash);
                     if(ctx.error.isZero()) {
                         ctx.error = sql_step.error;
                     }
                     */
                } catch (e) {

                }
            }

            sql_step.elapsed = ctx.getElapsedTime() - sql_step.start_time;
            ctx.sql_time += sql_step.elapsed;

            TraceSQL.isSlowSQL(ctx);

            // if(conf.getProperty('profile_sql_resource_enabled', false)){
            //     sql_step.start_cpu = ResourceProfile.getCPUTime();
            //     sql_step.start_mem = ResourceProfile.getUsedHeapSize();
            // }

            ctx.footprint('MySql Query Done');

            MeterSql.add(dbc_hash, sql_step.elapsed, false);
            StatSql.addSqlTime(ctx.service_hash, sql_step.dbc,
                sql_step.hash, sql_step.elapsed, args[0] != null, 0);

            if (Array.isArray(args[1]) && psql != null && psql.type === 'S') {
                var result_step = new ResultSetStep();
                result_step.start_time = ctx.getElapsedTime();
                result_step.elapsed = 0;
                result_step.fetch = args[1].length;
                result_step.sqlhash = psql.sql;
                result_step.dbc = dbc_hash;
                ctx.profile.push(result_step);

                ctx.rs_count = ctx.rs_count ? ctx.rs_count + args[1].length : args[1].length;
                ctx.rs_time = ctx.rs_time ? ctx.rs_time + sql_step.elapsed : sql_step.elapsed;
                // ctx.rs_time = ctx.rs_time ? ctx.rs_time + (result_step.start_time - ctx.getElapsedTime()) : result_step.start_time - ctx.getElapsedTime();

                MeterSql.addFetch(result_step.dbc, result_step.fetch, 0);
                StatSql.addFetch(result_step.dbc, result_step.sqlhash, result_step.fetch, 0);

                TraceSQL.isTooManyRecords(sql_step, result_step.fetch, ctx);
            }

            if (args[1] != null && psql != null && psql.type === 'U') {
                sql_step.updated = args[1].affectedRows || 0;
                //StatSql.addUpdate(dbc_hash, psql.sql, sql_step.updated);
            }
        }

        lastCallback = agent.aop.functionHook(args, -1, queryCallback);

        if (lastCallback === false && args.length > 0 && args[0]._callback) {
            agent.aop.both(args[0], '_callback', queryCallback);
        }
    }
};

var toParamBytes = function (p, crc) {
    if (p == null || p.length === 0) {
        return null;
    }
    try {
        return ParamSecurity.encrypt(Buffer.from(p, 'utf8'), crc);
    } catch (e) {
        return null;
    }
};

var errorDelegate = function () {
    return function (obj, args) {
        var ctx = TraceContextManager.getCurrentContext();
        if (ctx == null) {
            return;
        }

        var laststep = ctx.profile.getLastSteps(1);
        if (laststep == null || laststep.length === 0) {
            return;
        }

        var step = laststep[0];
        if (!args[0].fatal) {
            MeterSql.add(step.dbc, step.elapsed, true);
            StatSql.addSqlTime(ctx, ctx.service_hash, step.dbc, step.hash, step.elapsed, true, 0);
        }

        try {
            if (args[0].fatal) {
                step.error = StatError.addError('mysql-' + args[0].code, args[0].message, ctx.service_hash); /*long*/
                if (ctx.error.isZero()) {
                    ctx.error = step.error;
                }
            } else {
                if (conf._is_trace_ignore_err_cls_contains === true && args[0].code.indexOf(conf.trace_ignore_err_cls_contains) < 0) {
                    step.error = StatError.addError('mysql-' + args[0].code, args[0].message, ctx.service_hash, TextTypes.SQL, step.hash); /*long*/
                    if (ctx.error.isZero()) {
                        ctx.error = step.error;
                    }
                } else if (conf._is_trace_ignore_err_msg_contains === true && args[0].message.indexOf(conf.trace_ignore_err_msg_contains) < 0) {
                    step.error = StatError.addError('mysql-' + args[0].code, args[0].message, ctx.service_hash, TextTypes.SQL, step.hash); /*long*/
                    if (ctx.error.isZero()) {
                        ctx.error = step.error;
                    }
                } else if (conf._is_trace_ignore_err_cls_contains === false && conf._is_trace_ignore_err_msg_contains === false) {
                    step.error = StatError.addError('mysql-' + args[0].code, args[0].message, ctx.service_hash, TextTypes.SQL, step.hash); /*long*/
                    if (ctx.error.isZero()) {
                        ctx.error = step.error;
                    }
                }
            }
        } catch (e) {
        }
    }
};

MysqlObserver.prototype.inject = function (mod, moduleName) {
    if (mod.__whatap_observe__) {
        return;
    }
    mod.__whatap_observe__ = true;
    Logger.initPrint("MysqlObserver");
    var self = this;
    var aop = self.agent.aop;

    if (conf.sql_enabled === false) {
        return;
    }

    var dbc_hash = 0;
    var dbc = 'mysql://';
    aop.both(mod, 'createConnection',
        function (obj, args, lctx) {
            var ctx = lctx.context;
            if (ctx == null || ctx.db_opening) {
                return;
            }
            ctx.db_opening = true;
            ctx.footprint('MySql Connecting Start');

            var dbc_step = new DBCStep();
            dbc_step.start_time = ctx.getElapsedTime();
            lctx.step = dbc_step;
        },
        function (obj, args, ret, lctx) {

            if (dbc_hash === 0) {
                if (args.length > 0) {
                    var info = (args[0] || {});
                    dbc = 'mysql://';
                    dbc += info.user || '';
                    dbc += "@";
                    dbc += info.host || '';
                    dbc += '/';
                    dbc += info.database || '';
                    dbc_hash = HashUtil.hashFromString(dbc);
                    DataTextAgent.DBC.add(dbc_hash, dbc);
                    DataTextAgent.METHOD.add(dbc_hash, dbc);
                    DataTextAgent.ERROR.add(dbc_hash, dbc);
                }
            }

            aop.both(ret, 'query', queryHook(dbc, self.agent), function (obj, args, ret, lctx) {
                var ctx = lctx.context;
                TraceContextManager.resume(ctx);
            });

            aop.both(ret, 'execute', queryHook(dbc, self.agent), function (obj, args, ret, lctx) {
                var ctx = lctx.context;
                TraceContextManager.resume(ctx);
            });

            aop.after(ret._protocol, '_delegateError', errorDelegate());

            var ctx = lctx.context,
                step = lctx.step;

            if (ctx == null || step == null) {
                return;
            }

            TraceContextManager.resume(ctx);
            ctx.footprint('MySql Connecting Done');
            ctx.db_opening = false;
            step.hash = dbc_hash;

            aop.both(ret, 'connect', function (obj, args) {
                    var ctx = TraceContextManager.getCurrentContext();
                    if (ctx == null) {
                        return;
                    }

                    ctx.footprint('MySql Connecting Start');

                    ctx.db_opening = true;

                    aop.functionHook(args, -1, function (obj, args) {
                        if (args[0] && step.error.isZero()) {
                            step.error = StatError.addError('mysql-' + args[0].code,
                                args[0].message, ctx.service_hash);
                            step.elapsed = ctx.getElapsedTime() - step.start_time;
                        }
                        TraceContextManager.resume(ctx._id);
                    });
                    ctx.profile.push(step);
                }
                ,
                function (obj, args, ret, lctx) {
                    if (lctx.context) {
                        ctx.db_opening = false;
                        ctx.footprint('MySql Connecting Done');
                    }
                }
            );
        });


    var dbc_pool_hash = 0;
    aop.after(mod, ['createPool', 'createPoolCluster'], function (obj, args, ret) {
        if (dbc_pool_hash == 0 && dbc === 'mysql://') {
            if (args.length > 0) {
                var info = args[0];
                //Open [DatabaseName] connection 메세지 보여줄때 필요함.
                dbc += info.user || '';
                dbc += "@";
                dbc += info.host || '';
                dbc += '/';
                dbc += info.database || '';
            }

            // dbc_pool_hash = HashUtil.hashFromString(dbc);
            // DataTextAgent.DBC.add(dbc_pool_hash, dbc);
            // DataTextAgent.METHOD.add(dbc_pool_hash, dbc);
            // DataTextAgent.ERROR.add(dbc_pool_hash, dbc);
        }
        aop.both(ret, 'getConnection', function (obj, args, lctx) {
            var ctx = TraceContextManager.getCurrentContext();
            if (ctx == null) {
                return;
            }

            aop.functionHook(args, -1, function (obj, args) {
                TraceContextManager.resume(ctx._id);
                DataTextAgent.DBC.add(dbc_hash, dbc);

                if (args[0] != null) {
                    return;
                }
                var conn = args[1];
                if (conn.__query_hook__) {
                    return;
                }
                conn.__query_hook__ = true;
                aop.before(conn, 'query', queryHook(dbc, self.agent));
                aop.before(conn, 'execute', queryHook(dbc, self.agent));
                aop.before(conn._protocol, '_delegateError', errorDelegate(ctx));
            });
        }, function (obj, args, ret, lctx) {
            if (obj._allConnections != undefined) {
                var all = obj._allConnections.length;
                var idle = obj._freeConnections.length;
                MeterSql.setConnection((all - idle), idle, dbc);
            }
        });
    });
};

var checkedSql = new IntKeyMap(2000).setMax(2000);
var nonLiteSql = new IntKeyMap(5000).setMax(5000);
var date = DateUtil.yyyymmdd();

function escapeLiteral(sql) {
    if (sql == null) {
        sql = '';
    }

    if (date !== DateUtil.yyyymmdd()) {
        checkedSql.clear();
        nonLiteSql.clear();
        date = DateUtil.yyyymmdd();
        Logger.print('WHATAP-SQL-CLEAR', 'MysqlObserver CLEAR OK!!!!!!!!!!!!!!!!', false);
    }

    var sqlHash = HashUtil.hashFromString(sql);
    var psql = nonLiteSql.get(sqlHash);

    if (psql != null) {
        return psql;
    }
    psql = checkedSql.get(sqlHash);

    if (psql != null) {
        return psql;
    }

    var els = new EscapeLiteralSQL(sql);
    els.process();

    var hash = HashUtil.hashFromString(els.getParsedSql());
    DataTextAgent.SQL.add(hash, els.getParsedSql());

    if (hash === sqlHash) {
        psql = new ParsedSql(els.sqlType, hash, null);
        nonLiteSql.put(sqlHash, psql);
    } else {
        psql = new ParsedSql(els.sqlType, hash, els.getParameter());
        checkedSql.put(sqlHash, psql);
    }
    return psql;
}

exports.MysqlObserver = MysqlObserver;