/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var TraceContextManager = require('../trace/trace-context-manager'),
    SqlStepX            = require('../step/sql-stepx'),
    ResultSetStep       = require('../step/resultset-step'),
    DataTextAgent       = require('../data/datatext-agent'),
    StatError           = require('../stat/stat-error'),
    TextTypes           = require('../lang/text-types'),
    hashUtil            = require('../util/hashutil'),
    Logger              = require('../logger');

var MongodbObserver = function(agent){
    this.agent = agent;
    this.packages = ['mongodb'];
}

MongodbObserver.prototype.inject = function( mod, moduleName ) {
    var self = this;
    var aop = self.agent.aop;

    if('instrument' in mod == false) {
        return;
    }

    if(mod.__whatap_observe__) { return; }
    mod.__whatap_observe__ = true;

    Logger.initPrint("MongodbObserver");
    
    var requestTable = {},
        nextRequest,
        listener = mod.instrument({}, function(err, instrumentations) {});

    listener.on('started', function (event) {
        var requestId = event.requestId, request;
        var ctx = TraceContextManager.getCurrentContext();
        if(ctx == null) { return; }

        if(nextRequest) {
            requestTable[requestId] = { step : nextRequest.step, ctx : nextRequest.ctx };
            nextRequest.requestId = requestId;
            nextRequest = null;
        } else if(event.commandName === 'findandmodify') {
            var sql_step = new SqlStepX();
            sql_step.start_time = ctx.getElapsedTime();
            ctx.profile.add(sql_step);
            requestTable[requestId] = {step : sql_step, ctx : ctx};
        } else if( requestTable[requestId] == null ) {
            return;
        }

        var dbc = '',
            dbc_hash = 0;
        if(event.connectionId) {
            var conn = event.connectionId;
            dbc = 'mongodb://' + (conn.host || '') + ':' + (conn.port || '') + '/' + (event.databaseName || '');
            dbc_hash = hashUtil.hashFromString(dbc);
            DataTextAgent.DBC.add(dbc_hash, dbc);
        }
        request = requestTable[requestId];
        request.step.dbc = dbc_hash;

        var dbName = event.databaseName,
            collection = event.command[event.commandName],
            command = Object.keys(event.command).reduce(function (result, key) {
                if(key != event.commandName && key != 'filter' && key != 'documents') {
                    result[key] = event.command[key];
                }
                return result;
            }, {});
        command = JSON.stringify(command).slice(0, 1000);

        var ctx = request.ctx;
        ctx.sql_count++;

        var command_string = '', queryKey = '', sql ='', sql_hash = '', crud ='';
        if(event.commandName === 'find') {
            queryKey = Object.keys(event.command.filter).toString().slice(0, 1000);
            command_string = ',key='+queryKey + ',command=' + command;
            crud = 'S';
            ctx.sql_select++;
        } else if(event.commandName === 'insert') {
            queryKey = Object.keys(event.command.documents[0]).toString().slice(0, 1000);
            command_string = ',documents='+queryKey + ',command=' + command;
            crud = 'I';
            ctx.sql_insert++;
        } else if(event.commandName === 'delete') {
            command_string = ',command=' + command;
            crud = 'D';
            ctx.sql_delete++;
        } else if(event.commandName === 'findandmodify') {
            command_string = ',command=' + command;
            crud = 'U';
            ctx.sql_update++;
        } else if(event.commandName === 'update') {
            command_string = ',command=' + command;
            crud = 'U';
            ctx.sql_update++;
        } else {
            ctx.sql_others++;
        }

        sql = event.commandName + ' : databaseName=' + dbName +',collection=' + collection + command_string;
        sql_hash = hashUtil.hashFromString(sql);
        DataTextAgent.SQL.add(sql_hash, sql);
        request.step.hash = sql_hash;
        request.step.crud = crud.charCodeAt(0);
    });

    listener.on('succeeded', function (event) {
        var requestId = event.requestId,
            request = requestTable[event.requestId];
        
        if(request == null) {
            delete requestTable[requestId];
            return;
        }
        
        TraceContextManager.resume(request.ctx._id);
        request.step.elapsed = event.duration;
        request.ctx.sql_time += request.step.elapsed;

        var command = event.commandName || '';
        if(command === 'find' && event.reply) {
            var cursor = event.reply.cursor;
            if(cursor != null) {
                var fetch = cursor.firstBatch.length;
                var rs_step = new ResultSetStep();
                rs_step.elapsed = 0;
                rs_step.dbc = request.step.dbc;
                rs_step.sqlhash = request.step.hash;
                rs_step.fetch = fetch;
                request.ctx.profile.add(rs_step);
            }
        }

        if(command === 'update' && event.reply) {
            request.step.updated = event.reply.nModified;
        }

        if(command === 'findandmodify' && event.reply) {
            var lastErrorObject = event.reply.lastErrorObject;
            request.step.updated = lastErrorObject.n || 0;
        }

        delete requestTable[requestId];
    });

    listener.on('failed', function (event) {
        var requestId = event.requestId,
            request = requestTable[event.requestId];
        if(request == null) {
            delete requestTable[requestId];
            return;
        }
        TraceContextManager.resume(request.ctx._id);
        request.step.elapsed = event.duration;
        request.ctx.sql_time += request.step.elapsed;

        var msgObj = { 'class': event.failure.message, 'msg': event.failure.message };
        request.step.error = StatError.addError("mogodb", event.failure.message,
            request.ctx.service_hash,  TextTypes.SQL, request.step.hash);
        if (request.ctx.error.isZero()) {
            request.ctx.error = request.step.error;
        }
        
        delete requestTable[requestId];
    });

    ['cursor', 'insert', 'update', 'remove'].forEach( function(command) {
        aop.before(mod.Server.prototype, command, function (obj, args) {
            // Select but... DB name undefined return
            if(command === 'cursor' && !args[1].find) return;

            var ctx = TraceContextManager.getCurrentContext();
            if(ctx == null) { return; }

            var step = new SqlStepX();
            step.start_time = ctx.getElapsedTime();
            ctx.profile.add(step);

            var req = nextRequest = {step : step, ctx : ctx};
            aop.functionHook(args, -1, function (obj, args) {
                TraceContextManager.resume(ctx._id);
                var requestId = req.requestId;
                requestTable[requestId] = undefined;
            });
        });
    });
};

exports.MongodbObserver = MongodbObserver;

