/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */


var TraceContextManager = require('../trace/trace-context-manager'),
    SqlStepX            = require('../step/sql-stepx'),
    DBCStep             = require('../step/dbc-step'),
    ResultSetStep       = require('../step/resultset-step'),
    HashUtil            = require('../util/hashutil'),
    DataTextAgent       = require('../data/datatext-agent'),
    Logger              = require('../logger');

var MongoObserver = function (agent) {
    this.agent = agent;
    this.packages = ['mongodb'];
};

MongoObserver.prototype.inject = function( mod, moduleName ) {
    if(mod.__whatap_observe__) {return;}
    mod.__whatap_observe__ = true;

    Logger.initPrint("MongoObserver");

    var self = this;
    var conn_dbc_hash=0;
    var dbc = '';
    self.agent.aop.both(mod.MongoClient.prototype, ['connect'], function (obj, args, lctx) {
        var ctx = TraceContextManager.getCurrentContext();
        if(ctx == null) { return; }
        lctx.context = ctx;

        if(obj.s && obj.s.url){
            dbc = obj.s.url;
            conn_dbc_hash = HashUtil.hashFromString(dbc);
            DataTextAgent.DBC.add(conn_dbc_hash, dbc);
            DataTextAgent.METHOD.add(conn_dbc_hash, dbc);
            DataTextAgent.ERROR.add(conn_dbc_hash, dbc);
        }

        var dbc_step = new DBCStep();
        dbc_step.hash = conn_dbc_hash;
        dbc_step.start_time = ctx.getElapsedTime();

        lctx.step = dbc_step;
    }, function (obj, args, ret, lctx) {
        var pool = ret;
        var ctx  = lctx.context,
            step = lctx.step;

        if(ctx == null || step == null) { return; }
        step.elapsed = ctx.getElapsedTime() - step.start_time;
        ctx.profile.push(step);
    });

    var insertCommnad = ['insert', 'insertOne', 'insertMany'];
    insertCommnad.forEach(function (command) {
        self.agent.aop.before(mod.Collection.prototype, command, function (obj, args) {
            var ctx = TraceContextManager.getCurrentContext();
            if(ctx == null) { return; }

            ctx.footprint('Mongodb Command Start: ' + command );
            var sql_step;
            try {
                sql_step = new SqlStepX();
                var sql = command ;//+ ': ' + JSON.stringify(Object.keys(args[0]));
                if(args[0]){
                    sql += ' field=' + JSON.stringify(Object.keys(args[0]))
                }
                sql_step.hash = HashUtil.hashFromString(sql);
                sql_step.start_time = ctx.getElapsedTime();
                //sql_step.crud = 'I'.charCodeAt(0);
                DataTextAgent.SQL.add(sql_step.hash, sql);
                ctx.profile.push(sql_step);
            } catch(e) {
                Logger.printError("WHATAP-605", "Mongodb insert error", e, false);
                sql_step = null;
            }

            self.agent.aop.functionHook(args, -1, function (obj, args) {
                if(TraceContextManager.resume(ctx._id) == null || sql_step == null) { return; }

                ctx.footprint('Mongodb Command Done: ' + command );

                sql_step.elapsed = ctx.getElapsedTime() - sql_step.start_time;
            });
        });
    });

    var findCommand = ['find'];
    findCommand.forEach(function (command) {
        self.agent.aop.both(mod.Collection.prototype, command, function (obj, args, lctx) {
            var ctx = lctx.context;
            if(ctx == null) { return; }

            ctx.footprint('Mongodb Command Start: ' + command );
            var sql_step;
            try {
                if(dbc){
                    sql_step = new SqlStepX();
                    var sql = command ;//+ ': ' + JSON.stringify(Object.keys(args[0]));
                    if(args[0]){
                        sql += ' where=' + JSON.stringify(Object.keys(args[0]))
                    }
                    sql_step.hash = HashUtil.hashFromString(sql);
                    sql_step.start_time = ctx.getElapsedTime();
                    //sql_step.crud = 'S'.charCodeAt(0);
                    DataTextAgent.SQL.add(sql_step.hash, sql);
                    ctx.profile.push(sql_step);
                }

            } catch (e) {
                Logger.printError("WHATAP-606", "Mongodb find error", e, false);
                sql_step = null;
            }

            self.agent.aop.functionHook(args, -1, function (obj, args, ret) {
                if(TraceContextManager.resume(ctx._id) == null) { return; }

                ctx.footprint('Mongodb Command Done: ' + command );

                if(args[0] != null) {
                    //error
                }

                if(args[1]) {
                    self.agent.aop.before(args[1], 'toArray', function (obj, args) {

                        var result_step = new ResultSetStep();
                        result_step.start_time = ctx.getElapsedTime();
                        result_step.sqlhash = sql_step.hash;
                        ctx.profile.push(result_step);

                        ctx.footprint('Mongodb Fetch Start');

                        self.agent.aop.functionHook(args, -1, null, function (obj, args) {
                            if(TraceContextManager.resume(ctx._id) == null) { return; }

                            ctx.footprint('Mongodb Fetch Done');

                            if(args[0]) { return; }
                            result_step.elapsed = ctx.getElapsedTime() - result_step.start_time;
                            if(Array.isArray(args[1])) {
                                result_step.fetch = args[1].length;
                            }
                        });
                    });
                }

                sql_step.elapsed = ctx.getElapsedTime() - sql_step.start_time;
            });

        }, function (obj, args, ret, lctx) {
            var ctx = lctx.context;
            if(ctx == null) { return; }
            //Monk
            if(ret) {
                var sql_step = ctx.profile.getLastSteps(1)[0];
                if(sql_step) {
                    self.agent.aop.after(ret, 'toArray', function (obj, args) {
                        sql_step.elapsed = ctx.getElapsedTime() - sql_step.start_time;
                        ctx.footprint('Mongodb Command Done');

                        var result_step = new ResultSetStep();
                        result_step.start_time = ctx.getElapsedTime();
                        result_step.sqlhash = sql_step.hash;
                        ctx.profile.push(result_step);
                        ctx.footprint('Mongodb Fetch Start');

                        self.agent.aop.functionHook(args, -1, null, function (obj, args) {
                            if(TraceContextManager.resume(ctx._id) == null) { return; }

                            ctx.footprint('Mongodb Fetch Done');

                            if(args[0]) { return; }
                            result_step.elapsed = ctx.getElapsedTime() - result_step.start_time;
                            if(Array.isArray(args[1])) {
                                result_step.fetch = args[1].length;
                            }
                        });
                    });
                }
            }
        });
    });

    var updateCommand = ['update', 'updateOne', 'updateMany'];
    updateCommand.forEach(function (command) {
        self.agent.aop.both(mod.Collection.prototype, command, function (obj, args) {
            //args[0] where, args[1] set
            var ctx = TraceContextManager.getCurrentContext();
            if(ctx == null) {return;}

            ctx.footprint('Mongodb Command Start: '+command);
            var sql_step;
            try{
                sql_step = new SqlStepX();
                var sql = command ;//+ ': set=' + JSON.stringify(Object.keys(args[1]))
                if(args[0]){
                    sql += ' where=' + JSON.stringify(Object.keys(args[0]))
                }
                if(args[1]){
                    if(args[1]['$set']) sql += ', field=' + JSON.stringify(Object.keys(args[1]['$set']));
                    else sql += ' field =' + JSON.stringify(Object.keys(args[1]));
                }
                sql_step.hash = HashUtil.hashFromString(sql);
                sql_step.start_time = ctx.getElapsedTime();
                //sql_step.crud = 'U'.charCodeAt(0);
                DataTextAgent.SQL.add(sql_step.hash, sql);
                ctx.profile.push(sql_step);
            } catch(e) {
                Logger.printError("WHATAP-607", "Mongodb update error", e, false);
                sql_step = null;
            }

            self.agent.aop.functionHook(args, -1, null, function (obj, args) {
                if(TraceContextManager.resume(ctx._id) == null || sql_step == null) { return; }

                ctx.footprint('Mongodb Command Done: '+command);

                if(args[0] != null) {
                    //error
                }
                // if(args.length > 1) {
                //     var result = args[1];
                //     sql_step.updated = result.result['nModified'] || 0;
                // }

                sql_step.elapsed = ctx.getElapsedTime() - sql_step.start_time;
            });
        });
    });

    var deleteCommand = ['delete', 'deleteMany', 'deleteOne', 'remove', 'removeMany', 'removeOne'];
    deleteCommand.forEach(function (command) {
        self.agent.aop.both(mod.Collection.prototype, command, function (obj, args) {
            //args[0] where
            var ctx = TraceContextManager.getCurrentContext();
            if(ctx == null) { return; }

            ctx.footprint('Mongodb Command Start: '+command);
            var sql_step;
            try{
                sql_step = new SqlStepX();
                var sql = command ;//+ ': ' + JSON.stringify(Object.keys(args[0]));
                if(args[0]){
                    sql += ', where=' + JSON.stringify(Object.keys(args[0]));
                }
                sql_step.hash = HashUtil.hashFromString(sql);
                sql_step.start_time = ctx.getElapsedTime();
                //sql_step.crud = 'D'.charCodeAt(0);
                DataTextAgent.SQL.add(sql_step.hash, sql);

                ctx.profile.push(sql_step);
            } catch(e) {
                Logger.printError("WHATAP-608", "Mongodb delete error", e, false);
                sql_step = null;
            }

            self.agent.aop.functionHook(args, -1, null, function (obj, args) {
                if(TraceContextManager.resume(ctx._id) == null || sql_step == null) { return; }

                ctx.footprint('Mongodb Command Done: '+command);

                if(args[0] != null) {
                    //error
                }
                sql_step.elapsed = ctx.getElapsedTime() - sql_step.start_time;
            });
        });
    });

    var anotherCommand = ['findAndReplace', 'findAndUpdate', 'findAndDelete'
        , 'findOneAndUpdate', 'findOneAndReplace', 'findOneAndDelete', 'findAndModify'];
    anotherCommand.forEach(function (command) {
        self.agent.aop.both(mod.Collection.prototype, command, function (obj, args) {
            var ctx = TraceContextManager.getCurrentContext();
            if(ctx == null) { return; }

            ctx.footprint('Mongodb Command Start: '+command);
            var sql_step;
            try{

                sql_step = new SqlStepX();
                var sql = 'mongodb ' + command ;//+ ': where=' + JSON.stringify(Object.keys(args[0]));
                sql_step.hash = HashUtil.hashFromString(sql);
                sql_step.start_time = ctx.getElapsedTime();
                // sql_step.crud = 'S'.charCodeAt(0);
                DataTextAgent.SQL.add(sql_step.hash, sql);
                ctx.profile.push(sql_step);
            } catch(e) {
                Logger.printError("WHATAP-609", "Mongodb delete error", e, false);
                sql_step = null;
            }

            self.agent.aop.functionHook(args, -1, null, function (obj, args) {
                if(TraceContextManager.resume(ctx._id) == null || sql_step == null) { return; }

                ctx.footprint('Mongodb Command Done: '+command);

                if(args[0] != null) {
                    //error
                }
                sql_step.elapsed = ctx.getElapsedTime() - sql_step.start_time;
            });
        });
    });
};

exports.MongoObserver = MongoObserver;