/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var TraceContextManager = require('../trace/trace-context-manager'),
    ParsedSql           = require('../trace/parsed-sql'),
    SqlStepX            = require('../step/sql-stepx'),
    DBCStep             = require('../step/dbc-step'),
    ResultSetStep       = require('../step/resultset-step'),
    DataTextAgent       = require('../data/datatext-agent'),
    StatSql             = require('../stat/stat-sql'),
    MeterSql            = require('../counter/meter/meter-sql'),
    conf                = require('../conf/configure'),
    IntKeyMap           = require('../util/intkey-map'),
    EscapeLiteralSQL    = require('../util/escape-literal-sql'),
    HashUtil            = require('../util/hashutil'),
    StatError           = require('../stat/stat-error'),
    TextTypes           = require('../lang/text-types'),
    ParamSecurity       = require('../util/paramsecurity'),
    Logger              = require('../logger'),
    conf                = require('../conf/configure'),
    DateUtil            = require('../util/dateutil'),
    Buffer              = require('buffer').Buffer,
    shimmer             = require('../core/shimmer'),
    TraceSQL            = require('../trace/trace-sql');

var MariaObserver = function (agent) {
    this.agent = agent;
    this.packages = ['mariadb'];
};

var dbc, dbc_hash;
MariaObserver.prototype.inject = function (mod, moduleName) {
    if (mod.__whatap_observe__) { return; }
    mod.__whatap_observe__ = true;

    Logger.initPrint("MariaObserver");

    if (!conf.sql_enabled) return;

    // shimmer.wrap(mod, 'createConnection', wrapCreateConnection(mod));
    shimmer.wrap(mod, 'createConnection', wrapConnection(mod));
    shimmer.wrap(mod, 'createPool', wrapCreatePool(mod));
};

var _finishQuery = function (ctx, sql_step) {
    sql_step.elapsed = ctx.getElapsedTime() - sql_step.start_time;
    TraceSQL.isSlowSQL(ctx);

    MeterSql.add(sql_step.hash, sql_step.elapsed, false);
    StatSql.addSqlTime(ctx, ctx.service_hash, sql_step.dbc, sql_step.hash, sql_step.elapsed, false, 0);
};

var _handleError = function (ctx, sql_step, err) {
    sql_step.elapsed = ctx.getElapsedTime() - sql_step.start_time;
    TraceSQL.isSlowSQL(ctx);

    MeterSql.add(sql_step.hash, sql_step.elapsed, false);
    StatSql.addSqlTime(ctx, ctx.service_hash, sql_step.dbc, sql_step.hash, sql_step.elapsed, true, 0);
};

function wrapCreateConnection(agent) {
    return function (original) {
        return async function (...args) {
            const ctx = TraceContextManager.getCurrentContext();
            if (!ctx || ctx.db_opening) return original.apply(this, args);

            ctx.db_opening = true;
            ctx.footprint('Maria Connecting Start');

            // const dbc_step = new DBCStep();
            // dbc_step.start_time = ctx.getElapsedTime();

            const connection = await original.apply(this, args);
            wrapConnectionMethods(connection, agent);

            ctx.footprint('Maria Connecting Done');
            ctx.db_opening = false;

            getDBCHash(args);
            // dbc_step.hash = dbc_hash;
            // ctx.profile.push(dbc_step);

            return connection;
        };
    };
}

function wrapConnection(agent) {
    return function (original) {
        return function (...args) {
            const connectionPromise = original.apply(this, args);
            getDBCHash(args);

            return connectionPromise.then(connection => {
                shimmer.wrap(connection, 'query', createQueryHook(agent));
                return connection;
            });
        }
    }
}

function wrapCreatePool(agent) {
    return function (original) {
        return function (...args) {
            const pool = original.apply(this, args);
            getDBCHash(args);

            shimmer.wrap(pool, 'getConnection', (originalGetConnection) => {
                return async function (...connArgs) {
                    const connection = await originalGetConnection.apply(this, connArgs);
                    wrapConnectionMethods(connection, agent);
                    return connection;
                };
            });

            return pool;
        };
    };
}

function wrapConnectionMethods(connection, agent) {
    shimmer.wrap(connection, 'query', createQueryHook(agent));
    shimmer.wrap(connection, 'execute', createQueryHook(agent));
}

function createQueryHook(agent) {
    return function (original) {
        return function (...args) {
            const ctx = TraceContextManager.getCurrentContext();
            if (!ctx || !args[0]) return original.apply(this, args);

            ctx.db_opening = true;
            ctx.footprint('Maria Connecting Start');

            const dbc_step = new DBCStep();
            ctx.footprint('Maria Connecting Done');
            ctx.db_opening = false;

            dbc_step.hash = dbc_hash;
            ctx.profile.push(dbc_step);

            const sql_step = new SqlStepX();
            sql_step.start_time = ctx.getElapsedTime();
            ctx.profile.push(sql_step);

            ctx.footprint('Maria Query Start');
            ctx.sql_count++;

            var sql = args.length > 0 ? args[0] : undefined,
                psql = null;

            if(typeof sql !== 'string') {
                sql = args[0].sql || undefined;
            }

            if (typeof sql === 'string' && sql.length > 0) {
                try {
                    psql = escapeLiteral(sql);
                } catch (e) {
                    Logger.printError('WHATAP-191', 'MariaObserver escapeliteral error', e);
                }
            } else {
                sql = '';
                psql = escapeLiteral(sql);
            }

            if(psql != null) {
                sql_step.hash = psql.sql;
                // sql_step.crud = psql.type.charCodeAt(0);
            }

            var els = new EscapeLiteralSQL(sql);
            els.process();

            ctx.active_sqlhash = sql_step.hash;

            if(conf.profile_sql_param_enabled) {
                var params = args.length > 1 && Array.isArray(args[1]) ? args[1] : undefined;
                sql_step.setTrue(1);
                var crc = {value : 0};
                sql_step.p1 = toParamBytes(psql.param, crc);
                if(params != undefined) {
                    const result = params.map((param) => {
                        if(typeof param === 'string'){
                            return `'${param}'`
                        }
                        return param
                    }).toString()
                    sql_step.p2 = toParamBytes(result, crc);
                }
                sql_step.pcrc = crc.value;
            }

            const result = original.apply(this, args);

            return result.then(res => {
                if (Array.isArray(res) && psql && psql.type === "S") {
                    var result_step = new ResultSetStep();
                    result_step.start_time = ctx.getElapsedTime();
                    result_step.elapsed = 0;
                    result_step.fetch = res.length;
                    result_step.sqlhash = psql.sql;
                    result_step.dbc = dbc_hash;
                    ctx.profile.push(result_step);

                    ctx.rs_count = ctx.rs_count ? ctx.rs_count + res.length : res.length;

                    MeterSql.addFetch(result_step.dbc, result_step.fetch, 0);
                    StatSql.addFetch(result_step.dbc, result_step.sqlhash, result_step.fetch, 0);

                    TraceSQL.isTooManyRecords(sql_step, result_step.fetch, ctx);
                }
                var isSelectQuery = psql && psql.type === "S"? true : false;
                _finishQuery(ctx, sql_step, isSelectQuery, res);
                return res;
            }).catch(err => {
                _handleError(ctx, sql_step, err)
                if (conf.trace_sql_error_stack && conf.trace_sql_error_depth) {
                    var traceDepth = conf.trace_sql_error_depth;

                    var errorStack = err.stack.split("\n");
                    if (errorStack.length > traceDepth) {
                        errorStack = errorStack.slice(0, traceDepth + 1);
                    }
                    ctx.error_message = errorStack.join("\n");
                    sql_step.error = ctx.error = StatError.addError('pgsql -' + err.code, err.message, ctx.service_hash, TextTypes.SQL, null);
                }
                throw err;
            });
        };
    };
}

var toParamBytes = function (p, crc) {
    if (p == null || p.length === 0) {
        return null;
    }
    try {
        return ParamSecurity.encrypt(Buffer.from(p, 'utf8'), crc);
    } catch (e) {
        return null;
    }
};

function getDBCHash(args) {
    dbc = `mariadb://${(args[0] || {}).user || ''}@${(args[0] || {}).host || ''}/${(args[0] || {}).database || ''}`;
    dbc_hash = HashUtil.hashFromString(dbc);
}

var checkedSql = new IntKeyMap(2000).setMax(2000);
var nonLiteSql = new IntKeyMap(5000).setMax(5000);
var date = DateUtil.yyyymmdd();

function escapeLiteral(sql) {
    if(sql == null) { sql = ''; }

    if(date !== DateUtil.yyyymmdd()) {
        checkedSql.clear();
        nonLiteSql.clear();
        date = DateUtil.yyyymmdd();
        Logger.print('WHATAP-SQL-CLEAR', 'MariaObserver CLEAR OK!!!!!!!!!!!!!!!!', false);
    }

    var sqlHash = HashUtil.hashFromString(sql);
    var psql = nonLiteSql.get(sqlHash);

    if(psql != null) {
        return psql;
    }
    psql = checkedSql.get(sqlHash);

    if(psql != null) {
        return psql;
    }

    var els = new EscapeLiteralSQL(sql);
    els.process();

    var hash = HashUtil.hashFromString(els.getParsedSql());
    DataTextAgent.SQL.add(hash, els.getParsedSql());

    if(hash === sqlHash) {
        psql = new ParsedSql(els.sqlType, hash, null);
        nonLiteSql.put(sqlHash, psql);
    } else {
        psql = new ParsedSql(els.sqlType, hash, els.getParameter());
        checkedSql.put(sqlHash, psql);
    }
    return psql;
}

exports.MariaObserver = MariaObserver;