/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

const { reqlog_x_txid } = require('../conf/config-default');
const controlHandler = require('../control/control-handler');
const RequestLog = require('../requestlog');
var TraceContextManager = require('../trace/trace-context-manager'),
    URLPatternDetector  = require('../trace/serviceurl-pattern-detector').Detector,
    DataTextAgent       = require('../data/datatext-agent'),
    DataProfileAgent    = require('../data/dataprofile-agent'),
    EventLevel          = require('../data/event-level'),
    HttpStepX           = require('../step/http-stepx'),
    TxRecord            = require('../service/tx-record'),
    StatError           = require('../stat/stat-error'),
    StatHttpc           = require('../stat/stat-httpc'),
    StatTranxMtCaller   = require('../stat/stat-tx-caller'),
    ProfilePack         = require('../pack/profile-pack'),
    DataInputX          = require('../io/data-inputx'),
    SecurityMaster      = require('../net/security-master'),
    MeterService        = require('../counter/meter/meter-service').MeterService,
    MeterUsers          = require('../counter/meter/meter-users'),
    MeterHttpC          = require('../counter/meter/meter-httpc'),
    conf                = require('../conf/configure'),
    HashUtil            = require('../util/hashutil'),
    UserIdUtil          = require('../util/userid-util'),
    DateUtil            = require('../util/dateutil'),
    IPUtil              = require('../util/iputil'),
    Hexa32              = require('../util/hexa32'),
    ResourceProfile     = require('../util/resourceprofile'),
    TextTypes           = require('../lang/text-types'),
    MessageStep         = require('../step/message-step'),
    SecureMsgStep = require('../step/securemsg-step'),
    PluginLoaderManager = require('../plugin/plugin-loadermanager'),
    Long                = require('long'),
    KeyGen              = require('../util/keygen'),
    Logger              = require('../logger'),
    IntKeyLinkedMap = require("../util/intkey-linkedmap");
const ParamSecurity = require("../util/paramsecurity");
const {Buffer} = require("buffer");
const shimmer = require('../core/shimmer');
const TraceHttpc = require('../trace/trace-httpc');

var _exts=new Set([".css",".js",".png", ".htm", ".html", ".gif", ".jpg", ".css", ".txt", ".ico"]);

var configIpHeaderKey = conf.getProperty('trace_http_client_ip_header_key', 'x-forwarded-for');
var configUserAgentKey = conf.getProperty('trace_user_agent_header_key', '');
var configRefererKey = conf.getProperty('trace_referer_header_key', '');
var trace_origin_url = conf.getProperty('trace_origin_url', false);
var ignore_http_method = conf.getProperty('ignore_http_method', 'PATCH,OPTIONS,HEAD,TRACE');
var transaction_status_error_enable = conf.getProperty('transaction_status_error_enable', true);
var status_ignore = conf.getProperty('status_ignore', '');
var httpc_status_ignore = conf.getProperty('httpc_status_ignore', '');
var status_ignore_set = conf.getProperty('status_ignore_set', '');
var httpc_status_ignore_set = conf.getProperty('httpc_status_ignore_set', '');
var profile_error_step_enabled = conf.getProperty('profile_error_step_enabled', '');
var httpc_not_found_ignore = conf.getProperty('httpc_not_found_ignore', false);
var httpc_not_found_ignore_time = conf.getProperty('httpc_not_found_ignore_time', 300000);
var profile_http_header_ignore_keys = conf.getProperty('profile_http_header_ignore_keys', 'Cookie,cookie,accept,user-agent,referer');
var profile_http_parameter_enabled = conf.getProperty('profile_http_parameter_enabled', true);
var profile_http_parameter_keys = conf.getProperty('profile_http_parameter_keys', '');
var ignore_build_file_enabled = conf.getProperty('ignore_build_file_enabled', true);
var ignore_build_file_path = conf.getProperty('ignore_build_file_path', '/_next/');
conf.on('trace_http_client_ip_header_key', function(newProperty) {
    configIpHeaderKey = newProperty;
});
conf.on('trace_normalize_urls', function (newProps) {
    URLPatternDetector.build(true);
});
conf.on('trace_auto_normalize_enabled', function (newProps) {
    URLPatternDetector.build(true);
});
conf.on('trace_normalize_enabled', function (newProps) {
    URLPatternDetector.build(true);
});
conf.on('trace_user_agent_header_key', function (newProps) {
    configUserAgentKey = newProps;
});
conf.on('trace_referer_header_key', function (newProps) {
    configRefererKey = newProps;
});
conf.on('trace_origin_url', function (newProps) {
    trace_origin_url = newProps;
})
conf.on('ignore_http_method', function (newProps) {
    ignore_http_method = newProps;
})
conf.on('transaction_status_error_enable', function (newProps) {
    transaction_status_error_enable = newProps;
})
conf.on('status_ignore', function (newProps) {
    status_ignore = !newProps ? "" : String(newProps);
})
conf.on('httpc_status_ignore', function (newProps) {
    httpc_status_ignore = !newProps ? "" : String(newProps);
})
conf.on('status_ignore_set', function (newProps) {
    status_ignore_set = newProps;
})
conf.on('httpc_status_ignore_set', function (newProps) {
    httpc_status_ignore_set = newProps;
})
conf.on('profile_error_step_enabled', function (newProps) {
    profile_error_step_enabled = newProps;
})
conf.on('httpc_not_found_ignore', function (newProps) {
    httpc_not_found_ignore = newProps;
})
conf.on('httpc_not_found_ignore_time', function (newProps) {
    httpc_not_found_ignore_time = newProps;
})
conf.on('profile_http_header_ignore_keys', function (newProps) {
    profile_http_header_ignore_keys = newProps;
})
conf.on('profile_http_parameter_enabled', function (newProps) {
    profile_http_parameter_enabled = newProps;
})
conf.on('profile_http_parameter_keys', function (newProps) {
    profile_http_parameter_keys = newProps;
})
conf.on('ignore_build_file_enabled', function (newProps) {
    ignore_build_file_enabled = newProps;
})
conf.on('ignore_build_file_path', function (newProps) {
    ignore_build_file_path = newProps;
})
var staticConents = function (newProps) {
    var x=new Set();
    var words = !newProps?[]:newProps.split(',');
    for(var i = 0 ; i < words.length ; i++) {
        var ex = words[i].trim();
        if(ex.length>0){
            if(ex.startsWith(".")){
                x.add(ex);
            }else{
                x.add("."+ex );
            }
        }
    }

    _exts =x;
};
conf.on('web_static_content_extensions', staticConents);
staticConents(conf["web_static_content_extensions"]);

var httpc_not_found_ignore_map = null;
var httpc_not_found_ignore_url = new Set();

var HttpObserver = function(agent){
    this.agent = agent;
    this.packages = ['http','https'];
    URLPatternDetector.build(true);
};

HttpObserver.prototype.__createTransactionObserver = function(callback, isHttps, server) {
    var self = this;
    var aop = this.agent.aop;

    return function (req, res) {
        TraceContextManager._asyncLocalStorage.run(initCtx(req, res), () => {
            var ctx = TraceContextManager._asyncLocalStorage.getStore();
            if(!ctx) {
                return callback(req, res);
            }

            PluginLoaderManager.do('httpservicestart', ctx, req, res);
            if(trace_origin_url === true){
                var originUrlHash = HashUtil.hashFromString('Origin url');
                var step = new MessageStep();
                step.hash = originUrlHash;
                step.start_time = ctx.getElapsedTime();
                // step.desc = req.url;

                DataTextAgent.MESSAGE.add(originUrlHash, 'Origin url');
                ctx.profile.add(step);
            }

            aop.after(res, 'end', function(obj, args) {
                if(ctx == null) { return; }
                PluginLoaderManager.do('httpserviceend', ctx, req, res);

                var not_found_ignore = httpc_not_found_ignore_url.has(req.url);
                if (httpc_not_found_ignore && !not_found_ignore && obj.statusCode === 404) {
                    const url_hash = HashUtil.hashFromString(req.url);
                    const currentTime = new Date().getTime();

                    if (!httpc_not_found_ignore_map) {
                        httpc_not_found_ignore_map = new IntKeyLinkedMap(500, 1).setMax(500);
                    }
                    updateNotFoundIgnoreMap(req.url, url_hash, currentTime);
                }

                ctx.isStaticContents = isStatic(req.url);
                ctx.http_method = req.method;
                if(conf.profile_http_querystring_enabled){
                    ctx.http_query = JSON.stringify(req.query);
                }
                ctx.http_content_type = (req.headers['content_type'] || '');
                ctx.status = Math.floor(obj.statusCode / 100);

                // 에러가 발생했지만 스택수집안했을 경우
                if (transaction_status_error_enable && ctx.status >= 4) {
                    if(ctx.error.isZero())
                        ctx.error = StatError.addError(obj.statusCode, obj.statusMessage, ctx.service_hash);
                    ctx.statusCode = obj.statusCode;
                    ctx.statusTitle = obj.statusMessage;
                    ctx.statusMessage = ctx.error_message;
                }

                var shouldEndTransaction = shouldEndCurrentTransaction(not_found_ignore, ctx, res, req.route ? req.route.path : '');
                if (shouldEndTransaction) {
                    self.__endTransaction(null, ctx, req, res);
                } else {
                    TraceContextManager.end(ctx._id);
                    ctx = null;
                }
            });

            try {
                return callback.apply(this, arguments);
            } catch (e) {
                Logger.printError("WHATAP-606", 'Hooking request failed..', e, false);
                self.__endTransaction(e, ctx, req, res);
                throw e;
            }
        });
    };
};

function isStatic(u){
    var x = u.lastIndexOf('.');
    if(x<=0) return false;

    var ext = u.substring(x);
    return _exts.has(ext);
}
function initCtx(req, res) {
    /*url이 없으면 추적하지 않는다*/
    if(!req.url) { return null; }
    if(ignore_http_method && ignore_http_method.toUpperCase().split(',').includes(req.method)) { return null; }
    if (ignore_build_file_enabled && ignore_build_file_path && ignore_build_file_path.split(',').some(path => req.url.startsWith(path))) {
        return null;
    }
    if( conf.getProperty('profile_enabled', true) === false ) {
        return null;
    }

    var ctx = TraceContextManager.start();
    if(ctx == null) { return null; }

    req.__ctx_id__ = ctx._id;

    var url = req.url,
        index = url.indexOf('?');
    if(index >= 0) {
        url = url.slice(0, index);
    }
    ctx.service_name = URLPatternDetector.normalize(url);
    try {
        if(conf.trace_service_port_enabled === true && server.address()) {
            ctx.service_name += ' {' + server.address().port + '}';
        }
    } catch(e){
    }

    try {
        if(conf.trace_transaction_name_header_key != null) {
            ctx.service_name = req.headers[conf.trace_transaction_name_header_key] + ctx.service_name;
        }
    } catch(e) {
    }

    ctx.service_hash = HashUtil.hashFromString(ctx.service_name);
    // DataTextAgent.SERVICE.add(ctx.service_hash, ctx.service_name);

    ctx.isStaticContents = isStatic(req.url);

    var referer = undefined;
    if(configRefererKey && req.headers[configRefererKey]){
        referer = req.headers[configRefererKey];
    }else if(req.headers.referer){
        referer = req.headers.referer;
    }
    if (referer) {
        ctx.referer = HashUtil.hashFromString(referer);
        DataTextAgent.REFERER.add(ctx.referer, referer);
    }

    if(configUserAgentKey && req.headers[configUserAgentKey]){
        ctx.userAgentString = req.headers[configUserAgentKey];
    }else if(req.headers['user-agent']){
        ctx.userAgentString = req.headers['user-agent'];
    }
    if(ctx.userAgentString){
        ctx.userAgent = HashUtil.hashFromString(ctx.userAgentString);
        DataTextAgent.USERAGENT.add(ctx.userAgent, ctx.userAgentString);
    }

    ctx.http_host=req.headers.host;
    if (ctx.http_host) {
        ctx.http_host_hash = HashUtil.hashFromString(ctx.http_host);
        DataTextAgent.HTTP_DOMAIN.add(ctx.http_host_hash, ctx.http_host);
    }

    ctx.originUrl = req.url;
    ctx.start_malloc = ResourceProfile.getUsedHeapSize();
    ctx.start_cpu = ResourceProfile.getCPUTime();
    ctx.http_method = req.method;
    if(conf.profile_http_querystring_enabled){
        ctx.http_query = JSON.stringify(req.query);
    }
    ctx.http_content_type = (req.headers['content_type'] || '');
    var remote_addr;
    try {
        remote_addr = req.headers[configIpHeaderKey] || req.connection.remoteAddress || "0.0.0.0";
        if(remote_addr.includes(',')) {
            remote_addr = remote_addr.split(',')[0].trim();
        }
        remote_addr=IPUtil.checkIp4(remote_addr);
        ctx.remoteIp = IPUtil.stringToInt(remote_addr);
        switch (conf.trace_user_using_type || 3) {
            case 1:
                ctx.userid = Long.fromNumber(ctx.remoteIp);
                MeterUsers.add(ctx.userid);
                break;
            case 2:
                // MeterUsers.add(ctx.userid);
                break;
            case 3:
                ctx.userid = UserIdUtil.getUserId(req, res, 0);
                MeterUsers.add(ctx.userid);
                break;
        }

    } catch (e) {
    }
    /************************************/
    /*   Header / param Trace           */
    /************************************/
    var header_enabled = false;
    if(conf.profile_http_header_enabled === true && req.headers) {
        header_enabled = true;
        var prefix = conf.profile_header_url_prefix;
        if(prefix && ctx.service_name.indexOf(prefix) < 0) {
            header_enabled = false;
        }
    }

    if(header_enabled) {
        var step = new MessageStep();
        step.hash = HashUtil.hashFromString("HTTP-HEADERS");
        step.start_time = ctx.getElapsedTime();

        var header_ignore_key_set = new Set();
        if(profile_http_header_ignore_keys) {
            header_ignore_key_set = new Set(profile_http_header_ignore_keys.split(','));
        }

        try{
            step.desc = Object.keys(req.headers).map(function (key) {
                if (!header_ignore_key_set.has(key)) {
                    return `${key}=${req.headers[key]}`;
                }
            }).filter(element => {
                if (element) return element
            }).join('\n');

            DataTextAgent.MESSAGE.add(step.hash, "HTTP-HEADERS");
            ctx.profile.push(step);
        }catch (e) {
            step = null;
            header_ignore_key_set = null;
            Logger.printError('WHATAP-614', 'Header parsing error', e, false);
        }
    }
    /************************************/
    /*   Multi Server Transaction Trace */
    /************************************/
    if(conf.getProperty('mtrace_enabled', false)) {
        var poid=req.headers['x-wtap-po'];
        if (poid != null) {
            ctx.setCallerPOID(poid);
            try{
                var mt_caller=req.headers[conf._trace_mtrace_caller_key];
                if(mt_caller){
                    var x = mt_caller.indexOf(',');
                    if (x > 0) {
                        ctx.mtid = Hexa32.toLong32(mt_caller.substring(0, x));
                        ctx.mtid_build_checked = true;
                        var y = mt_caller.indexOf(',', x + 1);
                        if (y > 0) {
                            ctx.mdepth = parseInt(mt_caller.substring(x + 1, y));
                            var z = mt_caller.indexOf(',', y + 1);
                            if (z < 0) {
                                ctx.mcaller_txid = Hexa32.toLong32(mt_caller.substring(y + 1));
                            } else {
                                ctx.mcaller_txid = Hexa32.toLong32(mt_caller.substring(y + 1, z));

                                var z2 = mt_caller.indexOf(',', z + 1);
                                if (z2 < 0) {
                                    ctx.mcaller_stepId = Hexa32.toLong32(mt_caller.substring(z + 1));
                                } else {
                                    ctx.mcaller_stepId = Hexa32.toLong32(mt_caller.substring(z + 1, z2));
                                }
                            }
                        }
                    }
                }
                if (conf.stat_mtrace_enabled) {
                    var inf=req.headers[conf._trace_mtrace_spec_key1];
                    if(inf != null && inf.length > 0){
                        var px = inf.indexOf(',');
                        ctx.mcaller_spec =  inf.substring(0, px);
                        ctx.mcaller_url = inf.substring(px + 1);
                    }
                }
            } catch(e) {
                Logger.printError('WHATAP-850', 'Multi Server Transaction ', e, true);
            }
        }
    }

    return ctx;
};

HttpObserver.prototype.__endTransaction = function(error, ctx, req, res) {
    var param_enabled = false;
    if(conf.profile_http_parameter_enabled === true) {
        param_enabled = true;
        var prefix = conf.profile_http_parameter_url_prefix;
        if(prefix && ctx.service_name.indexOf(prefix) < 0) {
            param_enabled = false;
        }
    }

    if (param_enabled && req.query && Object.keys(req.query).length > 0) {
        const query = req.query;
        let secureMsgStep = new SecureMsgStep(ctx.getElapsedTime());
        secureMsgStep.hash = HashUtil.hashFromString("HTTP-PARAMETERS");

        const profileHttpParameterKeysSet = profile_http_parameter_keys ? new Set(profile_http_parameter_keys.split(',')) : null;

        try {
            const desc = Object.entries(query)
                .filter(([key]) => !profileHttpParameterKeysSet || profileHttpParameterKeysSet.has(key))
                .map(([key, value]) => `${key}=${value}`)
                .join('\n');

            if (desc) {
                const crc = { value: 0 };
                secureMsgStep.value = toParamBytes(desc, crc);
                secureMsgStep.crc = crc.value;

                DataTextAgent.MESSAGE.add(secureMsgStep.hash, "HTTP-PARAMETERS");
                ctx.profile.push(secureMsgStep);
            } else {
                secureMsgStep = null;
            }
        } catch (e) {
            secureMsgStep = null;
            Logger.printError('WHATAP-613', 'Parameter parsing error', e, false);
        }
    }

    if(profile_error_step_enabled && ctx.statusMessage){
        var step = new MessageStep();
        var title = ctx.statusTitle ? ctx.statusTitle : "EXCEPTION";
        step.hash = HashUtil.hashFromString(title);
        step.start_time = ctx.getElapsedTime();
        step.desc = ctx.statusMessage;
        DataTextAgent.MESSAGE.add(step.hash, title);
        ctx.profile.push(step);
    }

    if(error) {
        TraceContextManager.end(ctx != null ? ctx._id : null);
        ctx = null;
        return;
    }

    if(ctx == null || TraceContextManager.isExist(ctx._id) === false) { return; }

    if(ctx.isStaticContents !== true && conf._trace_ignore_url_set[ctx.service_hash]){
        ctx.isStaticContents = true;
    }

    if(conf._is_trace_ignore_url_prefix === true && ctx.service_name.startsWith(conf.trace_ignore_url_prefix) === true){
        ctx.isStaticContents = true;
    }

    if(ctx.isStaticContents){
        TraceContextManager.end(ctx._id);
        ctx = null;
        return;
    }

    try {
        var profile = new ProfilePack();
        var wtx = new TxRecord();
        // profile.time = ctx.start_time;
        wtx.endTime = DateUtil.currentTime();
        profile.time  = wtx.endTime;
        wtx.elapsed = ctx.getElapsedTime();

        ctx.service_hash = HashUtil.hashFromString(ctx.service_name);
        DataTextAgent.SERVICE.add(ctx.service_hash, ctx.service_name);

        wtx.service = ctx.service_hash;
        wtx.cpuTime = ResourceProfile.getCPUTime() - ctx.start_cpu;
        wtx.malloc = ResourceProfile.getUsedHeapSize()-ctx.start_malloc;
        if(wtx.malloc < 0) { wtx.malloc = 0; }
        wtx.originUrl = ctx.originUrl;

        wtx.seq = ctx.txid;
        wtx.sqlCount = ctx.sql_count;
        wtx.sqlTime = ctx.sql_time;
        wtx.sqlFetchCount = ctx.rs_count;
        wtx.sqlFetchTime = parseInt(ctx.rs_time);
        wtx.ipaddr = ctx.remoteIp;
        wtx.userid = ctx.userid;

        if (ctx.error.isZero() === false) {
            wtx.error = ctx.error;
            wtx.errorLevel = EventLevel.WARNING;
        }
        wtx.userAgent = ctx.userAgent;
        wtx.referer = ctx.referer;

        wtx.httpcCount = ctx.httpc_count;
        wtx.httpcTime = ctx.httpc_time;
        wtx.status = ctx.status;

        wtx.http_method=TxRecord.HTTP_METHOD[ctx.http_method] || TxRecord.WEB_GET;
        wtx.http_host = ctx.http_host;
        wtx.mtid=ctx.mtid;
        wtx.mdepth=ctx.mdepth;
        wtx.mcaller=ctx.mcaller_txid;
        wtx.mcallerStepId = ctx.mcaller_stepId;
        wtx.mcaller_pcode = ctx.mcaller_pcode;
        wtx.mcaller_okind = ctx.mcaller_okind;
        wtx.mcaller_oid = ctx.mcaller_oid;

        MeterService.add(wtx.service, wtx.elapsed,
            wtx.errorLevel, ctx.mcaller_pcode, ctx.mcaller_okind, ctx.mcaller_oid);

        profile.oid = SecurityMaster.OID;
        profile.service = wtx;
        if(Boolean(conf.reqlog_enabled) == true) {
            ctx.endTime = wtx.endTime;
            ctx.elapsed = wtx.elapsed;
            RequestLog.setRecord(ctx);
        }
        //duplicated executed... so end() first
        TraceContextManager.end(ctx._id);
        setTimeout(function () {
            DataProfileAgent.sendProfile(ctx, profile, false);
            TraceContextManager.end(ctx._id);
            ctx = null;
        }, 100);

    } catch (e) {
        Logger.printError('WHATAP-607', 'End transaction error..', e, false);
        TraceContextManager.end(ctx._id);
        ctx = null;
    }

};

var transfer_poid;
function transferPOID(ctx) {
    if (transfer_poid)
        return transfer_poid;
    transfer_poid=Hexa32.toString32(SecurityMaster.PCODE)+','
        +Hexa32.toString32(SecurityMaster.OKIND)+','+Hexa32.toString32(SecurityMaster.OID);
    return transfer_poid;
}

function transferMTID_CALLERTX(ctx) {
    if (ctx.transfer_id)
        return ctx.transfer_id;
    var x = Hexa32.toString32(ctx.mtid) + ',' + (ctx.mdepth + 1) + ',' + Hexa32.toString32(ctx.txid);
    ctx.transfer_id = x;
    return ctx.transfer_id;
}
function transferSPEC_URL(ctx) {
    if (ctx.transfer_info)
        return ctx.transfer_info;
    var x =  conf.mtrace_spec + ',' + ctx.service_hash;
    ctx.transfer_info = x;
    return ctx.transfer_info;
}
var check_seq = 1;
function interTxTraceAutoOn(ctx) {
    if (conf.mtrace_enabled ==false || ctx.httpc_checked || ctx.mtid.isZero()===false)
        return;
    ctx.httpc_checked = true;
    if(conf.mtrace_rate>=100){
        ctx.mtid = KeyGen.next();
        return;
    }
    check_seq++;
    switch (Math.floor(conf.mtrace_rate / 10)) {
        case 10:
            ctx.mtid = KeyGen.next();
            break;
        case 9:
            if (check_seq % 10 !== 0)
                ctx.mtid = KeyGen.next();
            break;
        case 8:
            if (check_seq % 5 !== 0)
                ctx.mtid = KeyGen.next();
            break;
        case 7:
            if (check_seq % 4 !== 0)
                ctx.mtid = KeyGen.next();
            break;
        case 6:
            if (check_seq % 3 !== 0)
                ctx.mtid = KeyGen.next();
            break;
        case 5:
            if (check_seq % 2 === 0)
                ctx.mtid = KeyGen.next();
            break;
        case 4:
            if (check_seq % 3 === 0 || check_seq % 5 === 0)
                ctx.mtid = KeyGen.next();
            break;
        case 3:
            if (check_seq % 4 === 0 || check_seq % 5 === 0)
                ctx.mtid = KeyGen.next();
            break;
        case 2:
            if (check_seq % 5 === 0)
                ctx.mtid = KeyGen.next();
            break;
        case 1:
            if (check_seq % 10 === 0)
                ctx.mtid = KeyGen.next();
            break;
    }
}
HttpObserver.prototype.inject = function( mod, moduleName ) {
    var self = this;
    var aop = self.agent.aop;

    if(mod.__whatap_observe__) { return; }
    mod.__whatap_observe__ = true;
    Logger.initPrint("HttpObserver");
    if( conf.getProperty('profile_enabled', true) === false ) { return; }

    aop.after(mod, 'createServer', function (obj, args, ret) {
        aop.before(ret, 'listen', function (obj, args) {
            conf['whatap.port'] = (args[0] || 80);
        });
    });

    aop.before(mod.Server.prototype, ['on', 'addListener'], function (obj, args) {
        if(args[0] !== 'request') {return;}

        args[args.length-1] = self.__createTransactionObserver(args[args.length-1],
            moduleName === 'https', obj);
    });

    if( conf.getProperty('httpc_enabled', true) === false ) { return; }

    shimmer.wrap(mod, 'request', function (original) {
        return function wrappedRequest(options, callback) {
            var ctx = TraceContextManager.getCurrentContext();
            if(!ctx || (!options.host && !options.hostname)){
                return original.apply(this, arguments);
            }

            var isHttpRepeat = false;
            if(moduleName === 'https'){
                options.__isHttps = true;
            }
            if(moduleName === 'http' && options.__isHttps){
                isHttpRepeat = true;
            }

            var step = new HttpStepX();
            step.start_time = ctx.getElapsedTime();

            if(!isHttpRepeat){
                if(options.method === 'OPTION'){
                    return original.apply(this, arguments);
                }
                try{
                    interTxTraceAutoOn(ctx);
                    if(conf.getProperty('mtrace_enabled', false)){
                        if(options.headers){
                            options.headers['x-wtap-po'] = transferPOID(ctx);
                        }else{
                            options.headers = {'x-wtap-po': transferPOID(ctx)};
                        }
                        if(conf.stat_mtrace_enabled){
                            options.headers[conf._trace_mtrace_spec_key1]=transferSPEC_URL(ctx);
                        }
                        if(ctx.mtid.isZero()===false){
                            options.headers[conf._trace_mtrace_caller_key]=transferMTID_CALLERTX(ctx);
                        }
                    }

                    ctx.httpc_url = options.path || '/';
                    ctx.httpc_host = options.host || options.hostname || '';
                    ctx.httpc_port = options.port || -1;

                    ctx.footprint('Http Call Start');

                    if (ctx.httpc_port < 0) { ctx.httpc_port = 80 };
                }catch (e) {
                    Logger.printError('WHATAP-852', 'Http Repeat ', e, true);
                    return original.apply(this, arguments);
                }
            }else{
                return original.apply(this, arguments);
            }

            var wrappedCallback;
            if (typeof callback === 'function') {
                wrappedCallback = function(response) {
                    if (TraceContextManager.resume(ctx._id) === null) {
                        return callback.apply(this, arguments);
                    }

                    response.on('end', function() {
                        step.elapsed = ctx.getElapsedTime() - step.start_time;
                        step.url = HashUtil.hashFromString(ctx.httpc_url);
                        DataTextAgent.HTTPC_URL.add(step.url, ctx.httpc_url);
                        step.host = HashUtil.hashFromString(ctx.httpc_host);
                        DataTextAgent.HTTPC_HOST.add(step.host, ctx.httpc_host);
                        step.port = ctx.httpc_port;

                        if (response.statusCode >= 400 && transaction_status_error_enable) {
                            step.error = StatError.addError(response.statusCode, response.statusMessage,
                                ctx.service_hash, TextTypes.HTTPC_URL, step.url);
                            if (ctx.error.isZero()) {
                                ctx.error = step.error;
                                ctx.statusCode = response.statusCode;
                                ctx.statusMessage = response.statusMessage;
                            }
                        }

                        ctx.active_httpc_hash = step.url;
                        ctx.profile.push(step);
                        endHttpc(ctx, step);
                    });

                    return callback.apply(this, arguments);
                };
            }

            var req = original.apply(this, [options, wrappedCallback]);
            var httpc_url = options.path

            req.on('error', function (err) {
                if (TraceContextManager.resume(ctx._id) === null) { return; }
                const isIgnoreHttpc = shouldIgnoreError(
                    err.code,
                    httpc_url,
                    httpc_status_ignore,
                    httpc_status_ignore_set);

                if (!isIgnoreHttpc && transaction_status_error_enable && step.error.isZero()) {
                    ctx.is_httpc_error = true;

                    step.error = StatError.addError(
                        err.code,
                        err.message,
                        ctx.service_hash,
                        TextTypes.HTTPC_URL,
                        step.url
                    );

                    if (ctx.error.isZero()) {
                        ctx.error = step.error;
                        ctx.statusCode = err.code;
                        ctx.statusMessage = err.message;
                    }
                }

                endHttpc(ctx, step);
            });

            shimmer.wrap(req, 'write', function (original) {
                return function wrappedWrite() {
                    try {
                        if (conf.getProperty('profile_httpc_parameter_enabled', true)) {
                            if (arguments && arguments[0]) {
                                var bodyData;

                                if (typeof arguments[0] === 'object' && !(arguments[0] instanceof Buffer)) {
                                    bodyData = JSON.stringify(arguments[0]);
                                }
                                if (arguments[0] instanceof Buffer) {
                                    bodyData = arguments[0].toString('utf8');
                                }
                                if(bodyData){
                                    var step = new MessageStep();
                                    step.hash = HashUtil.hashFromString("HTTPC-REQUEST-BODY");
                                    step.start_time = ctx.getElapsedTime();
                                    step.desc = bodyData;
                                    DataTextAgent.MESSAGE.add(step.hash, "HTTPC-REQUEST-BODY");
                                    ctx.profile.push(step);
                                }
                            }
                        }
                    } catch (e) {
                        Logger.printError('WHATAP-615', 'HTTP Write hook failed', e, false);
                    }
                    return original.apply(this, arguments);
                }
            });

            return req;
        }
    });

    function endHttpc(ctx, step) {
        if(ctx == null || step == null) { return; }

        step.elapsed = ctx.getElapsedTime() - step.start_time;
        TraceHttpc.isSlowHttpc(ctx, step);

        ctx.httpc_count++;
        ctx.httpc_time += step.elapsed;

        MeterHttpC.add(step.host, step.elapsed, step.error.isZero()===false);
        StatHttpc.addHttpcTime(ctx.service_hash, step.url, step.host, step.port, step.elapsed, step.error.isZero()===false);
        ctx.footprint('Http Call Done');
    }
};

function shouldIgnoreError(statusCode, url, statusCodeIgnore, statusIgnoreSet) {
    if (statusCodeIgnore) {
        const ignoreCodes = statusCodeIgnore.split(',').map(code => code.trim());
        if (ignoreCodes.includes(String(statusCode))) {
            return true;
        }
    }

    if (statusIgnoreSet) {
        const ignoreSet = statusIgnoreSet.split(',').map(item => item.trim());
        const urlStatusCombo = `${url}:${statusCode}`;
        if (ignoreSet.includes(urlStatusCombo)) {
            return true;
        }
    }

    return false;
}

function updateNotFoundIgnoreMap(url, url_hash, currentTime) {
    var ignore_value = httpc_not_found_ignore_map.get(url_hash) || {
        count: 0, start_time: currentTime, last_time: currentTime, url: url
    };

    ignore_value.count++;
    ignore_value.last_time = currentTime;

    if ((ignore_value.last_time - ignore_value.start_time) > httpc_not_found_ignore_time) {
        ignore_value = { count: 1, start_time: currentTime, last_time: currentTime, url: url };
    }

    if (ignore_value.count >= 50) {
        httpc_not_found_ignore_url.add(url);
        httpc_not_found_ignore_map.remove(url_hash);
        Logger.print('WHATAP-610', "The ignore option for URL " + url + " is applied.", false);
    }

    httpc_not_found_ignore_map.put(url_hash, ignore_value);
}

function shouldEndCurrentTransaction(not_found_ignore, ctx, res, requestPath) {
    if (not_found_ignore) return false;
    const is_ignore = shouldIgnoreError(res.statusCode, requestPath, status_ignore, status_ignore_set);
    return !is_ignore;
}

var toParamBytes = function (p, crc) {
    if (p == null || p.length === 0) {
        return null;
    }
    try {
        return ParamSecurity.encrypt(Buffer.from(p, 'utf8'), crc);
    } catch (e) {
        return null;
    }
};

exports.HttpObserver = HttpObserver;