/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var TraceContextManager = require('../trace/trace-context-manager'),
    conf = require('../conf/configure'),
    Logger = require('../logger');
const { Detector: URLPatternDetector } = require("../trace/serviceurl-pattern-detector");
const HashUtil = require("../util/hashutil");
const DataTextAgent = require("../data/datatext-agent");
const ResourceProfile = require("../util/resourceprofile");
const ProfilePack = require('../pack/profile-pack');
const TxRecord = require('../service/tx-record');
const DateUtil = require('../util/dateutil');
const SecurityMaster = require('../net/security-master');
const DataProfileAgent = require("../data/dataprofile-agent");
const MessageStep = require("../step/message-step");
const MeterService = require('../counter/meter/meter-service').MeterService;

const shimmer = require('../core/shimmer');

var grpc_profile_enabled = conf.getProperty('grpc_profile_enabled', true);
var grpc_profile_stream_client_enabled = conf.getProperty('grpc_profile_stream_client_enabled', true);
var grpc_profile_stream_server_enabled = conf.getProperty('grpc_profile_stream_server_enabled', true);
var grpc_profile_ignore_method = conf.getProperty('grpc_profile_ignore_method', '');
var ignore_method_set = null;
conf.on('grpc_profile_enabled', function(newProperty) {
    grpc_profile_enabled = newProperty;
});
conf.on('grpc_profile_stream_client_enabled', function(newProperty) {
    grpc_profile_stream_client_enabled = newProperty;
});
conf.on('grpc_profile_stream_server_enabled', function(newProperty) {
    grpc_profile_stream_server_enabled = newProperty;
});
conf.on('grpc_profile_ignore_method', function(newProperty) {
    grpc_profile_ignore_method = newProperty;
});

var GRpcObserver = function(agent) {
    this.agent = agent;
    this.packages = ['@grpc/grpc-js'];
};

const types = {
    unary: 'unary',
    client_stream: 'clientStream',
    server_stream: 'serverStream',
    bidi: 'bidi'
};

GRpcObserver.prototype.inject = function(mod, moduleName) {
    if (mod.__whatap_observe__) {
        return;
    }
    mod.__whatap_observe__ = true;
    Logger.initPrint("GRpcObserver");

    if (grpc_profile_enabled) {
        shimmer.wrap(mod.Server.prototype, 'register', wrapRegister);
        shimmer.wrap(mod.Client.prototype, 'register', wrapRegister);
    }
};

function checkIgnoreMethod(ignore_method, method_name){
    try{
        if (ignore_method) {
            ignore_method_set = new Set(ignore_method.split(','));
        } else {
            ignore_method_set = null;
        }
        if (ignore_method_set && ignore_method_set.has(method_name)) {
            return true;
        }
    }catch (e) {
        Logger.printError('WHATAP-703', 'gRPC checkIgnoreMethod error: ' + e, false);
    }
    return false;
}

function wrapHandler(handler, methodName, type) {
    return function(call, callback) {
        var method_name = methodName.includes('/') ? methodName.substring(methodName.lastIndexOf('/')+1, methodName.length) : methodName
        if (!grpc_profile_enabled || checkIgnoreMethod(conf.getProperty('grpc_profile_ignore_method', ''), method_name)) {
            return handler.call(this, call, callback);
        }

        TraceContextManager._asyncLocalStorage.run(initCtx(), () => {
            try {
                var ctx = TraceContextManager._asyncLocalStorage.getStore();
                if (!ctx) {
                    return handler.call(this, call, callback);
                }

                ctx.service_name = methodName;
                ctx.service_hash = HashUtil.hashFromString(methodName);

                var step_type = new MessageStep();
                step_type.hash = HashUtil.hashFromString('Type');
                step_type.start_time = ctx.getElapsedTime();
                step_type.desc = type;
                DataTextAgent.MESSAGE.add(step_type.hash, "Type");
                ctx.profile.push(step_type);

                var step_method = new MessageStep();
                step_method.hash = HashUtil.hashFromString('Method');
                step_method.start_time = ctx.getElapsedTime();
                step_method.desc = method_name;
                DataTextAgent.MESSAGE.add(step_method.hash, "Method");
                ctx.profile.push(step_method);

                if(call.request && Object.keys(call.request).length > 0){
                    var step_param = new MessageStep();
                    step_param.hash = HashUtil.hashFromString('Parameter');
                    step_param.start_time = ctx.getElapsedTime();
                    step_param.desc = JSON.stringify(Object.keys(call.request));
                    DataTextAgent.MESSAGE.add(step_param.hash, "Parameter");
                    ctx.profile.push(step_param);
                }

                console.log(call.pendingStatus)

                ctx.grpc_method = method_name;

                function wrappedCallback(err, response, ctx) {

                    if (err) {
                        ctx.error = err.stack;
                        ctx.statusCode = 500;

                        var step_error = new MessageStep();
                        step_error.hash = HashUtil.hashFromString("EXCEPTION");
                        step_error.start_time = ctx.getElapsedTime();
                        step_error.desc = err.stack;
                        DataTextAgent.MESSAGE.add(step_error.hash, "EXCEPTION");
                        ctx.profile.push(step_error);
                    }

                    endTransaction(ctx);

                    if (typeof callback === 'function') {
                        return callback(err, response);
                    }
                }

                return handler.call(this, call, (err, response) => wrappedCallback(err, response, ctx));
                // return handler.call(this, call, (err, response) => {
                //     if (!err) {
                //         err = new Error('Exception error occured');
                //     }
                //     wrappedCallback(err, response, ctx);
            } catch (err) {
                Logger.printError('WHATAP-701', 'gRPC wrapHandler error: ' + err, false);

                ctx.error = err.stack;

                var step_error = new MessageStep();
                step_error.hash = HashUtil.hashFromString("EXCEPTION");
                step_error.start_time = ctx.getElapsedTime();
                step_error.desc = err.stack;
                DataTextAgent.MESSAGE.add(step_error.hash, "EXCEPTION");
                ctx.profile.push(step_error);

                endTransaction(ctx);

                return handler.call(this, call, callback);
            }
        });
    };
}

function wrapStreamHandler(handler, methodName, type) {
    return function(call, callback) {
        var method_name = methodName.includes('/') ? methodName.substring(methodName.lastIndexOf('/')+1, methodName.length) : methodName
        if (!grpc_profile_enabled || checkIgnoreMethod(conf.getProperty('grpc_profile_ignore_method', ''), method_name)) {
            return handler.call(this, call, callback);
        }

        switch (type) {
            case 'server_stream':
                if (!grpc_profile_stream_server_enabled) {
                    return handler.call(this, call, callback);
                }
                break;
            case 'client_stream':
                if (!grpc_profile_stream_client_enabled) {
                    return handler.call(this, call, callback);
                }
                break;
            case 'bidi':
                if (!grpc_profile_stream_server_enabled || !grpc_profile_stream_client_enabled) {
                    return handler.call(this, call, callback);
                }
                break;
        }

        TraceContextManager._asyncLocalStorage.run(initCtx(), () => {
            try {
                var ctx = TraceContextManager._asyncLocalStorage.getStore();
                if (!ctx) {
                    return handler.call(this, call, callback);
                }

                ctx.service_name = methodName;
                ctx.service_hash = HashUtil.hashFromString(methodName);

                var step_type = new MessageStep();
                step_type.hash = HashUtil.hashFromString('Type');
                step_type.start_time = ctx.getElapsedTime();
                step_type.desc = type;
                DataTextAgent.MESSAGE.add(step_type.hash, "Type");
                ctx.profile.push(step_type);

                var step_method = new MessageStep();
                step_method.hash = HashUtil.hashFromString('Method');
                step_method.start_time = ctx.getElapsedTime();
                step_method.desc = method_name;
                DataTextAgent.MESSAGE.add(step_method.hash, "Method");
                ctx.profile.push(step_method);

                ctx.grpc_method = method_name;

                const startTime = Date.now();

                call.on('end', () => {
                    const duration = Date.now() - startTime;
                    endTransaction(ctx);
                });

                call.once('cancelled', () => {
                    ctx.cancelled = true;
                    var step_cancel = new MessageStep();
                    step_cancel.hash = HashUtil.hashFromString("CANCELLED");
                    step_cancel.start_time = ctx.getElapsedTime();
                    step_cancel.desc = "Request cancelled";
                    DataTextAgent.MESSAGE.add(step_cancel.hash, "CANCELLED");
                    ctx.profile.push(step_cancel);
                    endTransaction(ctx);
                });

                call.on('error', (err) => {
                    ctx.error = err.stack;
                    ctx.statusCode = err.code || grpc.status.INTERNAL;

                    var step_error = new MessageStep();
                    step_error.hash = HashUtil.hashFromString("EXCEPTION");
                    step_error.start_time = ctx.getElapsedTime();
                    step_error.desc = err.stack;
                    DataTextAgent.MESSAGE.add(step_error.hash, "EXCEPTION");
                    ctx.profile.push(step_error);

                    endTransaction(ctx);
                });

                return handler.call(this, call);
            } catch (e) {
                Logger.printError('WHATAP-702', 'gRPC wrapStreamHandler error: ' + e, false);
                return handler.call(this, call);
            }
        });
    };
}

function wrapRegister(register) {
    return function(name, handler, serialize, deserialize, type) {
        if (typeof handler === 'function') {
            switch (type) {
                case types.client_stream:
                    handler = wrapStreamHandler(handler, name, 'clientStream');
                    break;
                case types.server_stream:
                    handler = wrapStreamHandler(handler, name, 'serverStream');
                    break;
                case types.bidi:
                    handler = wrapStreamHandler(handler, name, 'bidi');
                    break;
                case types.unary:
                default:
                    handler = wrapHandler(handler, name, 'unary');
            }
        }
        return register.call(this, name, handler, serialize, deserialize, type);
    };
}

function endTransaction(ctx) {
    try {
        var profile = new ProfilePack();
        var wtx = new TxRecord();
        wtx.endTime = DateUtil.currentTime();
        profile.time = wtx.endTime;
        wtx.elapsed = ctx.getElapsedTime();

        DataTextAgent.SERVICE.add(ctx.service_hash, ctx.service_name);

        wtx.seq = ctx.txid;
        wtx.service = ctx.service_hash;
        wtx.cpuTime = ResourceProfile.getCPUTime() - ctx.start_cpu;
        wtx.malloc = ResourceProfile.getUsedHeapSize() - ctx.start_malloc;
        if (wtx.malloc < 0) { wtx.malloc = 0; }
        wtx.status = 2;
        wtx.http_method = ctx.grpc_method;

        wtx.ipaddr = ctx.remoteIp;

        MeterService.add(wtx.service, wtx.elapsed,
            wtx.errorLevel, ctx.mcaller_pcode, ctx.mcaller_okind, ctx.mcaller_oid);

        profile.oid = SecurityMaster.OID;
        profile.service = wtx;

        TraceContextManager.end(ctx._id);

        setTimeout(function() {
            DataProfileAgent.sendProfile(ctx, profile, false);
            ctx = null;
        }, 100);
    } catch (e) {
        Logger.printError('WHATAP-615', 'Websocket end transaction error..', e, false);
        TraceContextManager.end(ctx._id);
        ctx = null;
    }
};

function initCtx() {
    const ctx = TraceContextManager.start();
    if (!ctx) { return; }

    ctx.start_malloc = ResourceProfile.getUsedHeapSize();
    ctx.start_cpu = ResourceProfile.getCPUTime();
    return ctx;
}

exports.GRpcObserver = GRpcObserver;