/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var TraceContextManager = require('../trace/trace-context-manager'),
    Logger = require('../logger');

var HttpStepX = require('../step/http-stepx');
const Hexa32 = require("../util/hexa32");
const SecurityMaster = require("../net/security-master");
const conf = require("../conf/configure");
const KeyGen = require("../util/keygen");
const HashUtil = require("../util/hashutil");
const DataTextAgent = require("../data/datatext-agent");
const shimmer = require('../core/shimmer');
const StatError = require("../stat/stat-error");
const TextTypes = require("../lang/text-types");
const URLPatternDetector  = require('../trace/serviceurl-pattern-detector').Detector;

var GlobalObserver = function (agent) {
    this.agent = agent;
    this.packages = ['global'];
}

var transaction_status_error_enable = conf.getProperty('transaction_status_error_enable', true);
conf.on('transaction_status_error_enable', function (newProps) {
    transaction_status_error_enable = newProps;
})

GlobalObserver.prototype.inject = function (mod, moduleName) {
    var self = this;

    ['setTimeout', 'setInterval', 'setImmediate'].forEach(function (funcName) {
        self.agent.aop.before(mod, funcName, function (obj, args) {
            if(args[args.length - 1] === 'whatap') { return; }

            var cached_id = TraceContextManager.getCurrentId();
            self.agent.aop.functionHook(args, 0, function (obj, args) {
                TraceContextManager.resume(cached_id);
                cached_id = null;
            });
        });
    });

    shimmer.wrap(mod, 'fetch', function(original) {
        return async function(...args) {
            var info = args[1] ? args[1] : {};
            var ctx = TraceContextManager._asyncLocalStorage.getStore();

            if (ctx) {
                interTxTraceAutoOn(ctx);

                if (conf.getProperty('mtrace_enabled', false)) {
                    addTraceHeaders(info, ctx);
                    args[1] = info;
                }
            }

            try {
                const response = await original.apply(this, args);
                handleResponse(ctx, args, response);
                return response;
            } catch (e) {
                handleError(ctx, args, e);
                throw e;
            }
        };
    });
};

function addTraceHeaders(info, ctx) {
    if (info.headers) {
        info.headers['x-wtap-po'] = transferPOID(ctx);
    } else {
        info.headers = {
            'x-wtap-po': transferPOID(ctx)
        };
    }
    if (conf.getProperty('stat_mtrace_enabled', false)) {
        info.headers[conf._trace_mtrace_spec_key1] = transferSPEC_URL(ctx);
    }
    if (ctx.mtid.isZero() === false) {
        info.headers[conf._trace_mtrace_caller_key] = transferMTID_CALLERTX(ctx);
    }
}

function handleResponse(ctx, args, response) {
    if (!ctx || !args[0]) return;

    const url = new URL(args[0]);
    setupContext(ctx, url);

    var step = createStep(ctx, url);

    if (!response.ok && transaction_status_error_enable) {
        recordError(ctx, step, response);
    }

    ctx.profile.push(step);
}

function handleError(ctx, args, error) {
    if (!ctx || !args[0]) return;

    const url = new URL(args[0]);
    setupContext(ctx, url);

    var step = createStep(ctx, url);

    if (transaction_status_error_enable) {
        recordError(ctx, step, error);
    }

    ctx.profile.push(step);
}

function setupContext(ctx, url) {
    ctx.httpc_host = url.hostname;
    ctx.httpc_url = url.pathname;
    ctx.httpc_port = url.port || (url.protocol === 'https:' ? 443 : 80);
}

function createStep(ctx, url) {
    var step = new HttpStepX();
    step.start_time = ctx.getElapsedTime();
    step.url = HashUtil.hashFromString(ctx.httpc_url);
    DataTextAgent.HTTPC_URL.add(step.url, ctx.httpc_url);
    step.host = HashUtil.hashFromString(ctx.httpc_host);
    DataTextAgent.HTTPC_HOST.add(step.host, ctx.httpc_host);
    step.port = ctx.httpc_port;
    return step;
}

function recordError(ctx, step, response) {
    if (step.error.isZero()) {
        var cleanUrl = ctx.httpc_url.split('?')[0];
        ctx.service_name = URLPatternDetector.normalize(cleanUrl);
        ctx.service_hash = HashUtil.hashFromString(ctx.service_name);

        const errorMessage = response.statusText || (response.cause ? response.cause.message : '') || response.message || 'Unknown error';
        step.error = StatError.addError(response.status || null, errorMessage, ctx.service_hash, TextTypes.HTTPC_URL, step.url);

        if (ctx.error.isZero()) {
            ctx.error = step.error;
            ctx.statusCode = response.status || null;
            ctx.statusMessage = errorMessage;
        }
    }
}

var transfer_poid;
function transferPOID(ctx) {
    if (transfer_poid)
        return transfer_poid;
    transfer_poid = Hexa32.toString32(SecurityMaster.PCODE) + ','
        + Hexa32.toString32(SecurityMaster.OKIND) + ',' + Hexa32.toString32(SecurityMaster.OID);
    return transfer_poid;
}

function transferMTID_CALLERTX(ctx) {
    if (ctx.transfer_id)
        return ctx.transfer_id;
    var x = Hexa32.toString32(ctx.mtid) + ',' + (ctx.mdepth + 1) + ',' + Hexa32.toString32(ctx.txid);
    ctx.transfer_id = x;
    return ctx.transfer_id;
}

function transferSPEC_URL(ctx) {
    if (ctx.transfer_info)
        return ctx.transfer_info;
    var x = conf.mtrace_spec + ',' + ctx.service_hash;
    ctx.transfer_info = x;
    return ctx.transfer_info;
}

var check_seq = 1;
function interTxTraceAutoOn(ctx) {
    if (conf.mtrace_enabled == false || ctx.httpc_checked || ctx.mtid.isZero() === false)
        return;
    ctx.httpc_checked = true;
    if (conf.mtrace_rate >= 100) {
        ctx.mtid = KeyGen.next();
        return;
    }
    check_seq++;
    switch (Math.floor(conf.mtrace_rate / 10)) {
        case 10:
            ctx.mtid = KeyGen.next();
            break;
        case 9:
            if (check_seq % 10 !== 0)
                ctx.mtid = KeyGen.next();
            break;
        case 8:
            if (check_seq % 5 !== 0)
                ctx.mtid = KeyGen.next();
            break;
        case 7:
            if (check_seq % 4 !== 0)
                ctx.mtid = KeyGen.next();
            break;
        case 6:
            if (check_seq % 3 !== 0)
                ctx.mtid = KeyGen.next();
            break;
        case 5:
            if (check_seq % 2 === 0)
                ctx.mtid = KeyGen.next();
            break;
        case 4:
            if (check_seq % 3 === 0 || check_seq % 5 === 0)
                ctx.mtid = KeyGen.next();
            break;
        case 3:
            if (check_seq % 4 === 0 || check_seq % 5 === 0)
                ctx.mtid = KeyGen.next();
            break;
        case 2:
            if (check_seq % 5 === 0)
                ctx.mtid = KeyGen.next();
            break;
        case 1:
            if (check_seq % 10 === 0)
                ctx.mtid = KeyGen.next();
            break;
    }
}

exports.GlobalObserver = GlobalObserver;