/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var fs = require('fs'),
    path = require('path');

var TraceContextManager = require('../trace/trace-context-manager'),
    DataTextAgent       = require('../data/datatext-agent'),
    MessageStep         = require('../step/message-step'),
    StatError           = require('../stat/stat-error'),
    TextTypes           = require('../lang/text-types'),
    Configure           = require('../conf/configure'),
    HashUtil            = require('../util/hashutil'),
    conf                = require('../conf/configure'),
    Logger              = require('../logger');

var FileObserve = function(agent){
    this.agent = agent;
    this.packages = ['fs'];
};

var ingFile = {};
var fileDescriptors = {};

FileObserve.prototype.inject = function( mod ) {

    if(mod.__whatap_observe__) { return; }
    mod.__whatap_observe__ = true;
    Logger.initPrint("FileObserve");
    var extension = Configure.getProperty('web_static_content_extensions'),
        extensions = [],
        plugins = ['appservicestart', 'appserviceend', 'httpservicestart', 'httpserviceend', 'httpcall'];

    if(extension) {
        extensions = extension.split(',');
    }

    for(var i = 0 ; i < extensions.length ; i++) {
        var ex = extensions[i];
        extensions[i] = '.' + ex.trim();
    }

    var methods = [
        'rename',
        'truncate',
        'chown',
        'lchown',
        'fchown',
        'chmod',
        'lchmod',
        'fchmod',
        // 'stat',
        // 'lstat',
        // 'fstat',
        'link',
        'symlink',
        'readlink',
        'realpath',
        'unlink',
        'rmdir',
        'mkdir',
        'mkdtemp',
        'readdir',
        'close',
        'open',
        'utimes',
        'futimes',
        'fsync',
        'readFile',
        'writeFile',
        'appendFile',
        'exists',
        'ftruncate'
    ];

    var self = this,
        aop = self.agent.aop;

    var openFiles = {},
        fs = mod;

    methods.forEach(function (funcName) {
        aop.before(mod, funcName, function (obj, args) {
            if(args.length === 0) { return; }

            var fileName = args[0];

            if(typeof fileName != 'string') { return; }
            if(typeof conf['app.root'] != 'string') { return; }
            if(fileName.indexOf('/proc/') == 0 ){ return ; } // /proc/uptime & /proc/{pid}/stat 이 찍힌다.

            if( (fileName[0] && fileName[0] === '/') && (fileName[1] && fileName[1] === 'p') ){ return; }

            if(path.dirname(fileName) === path.join(conf['app.root'], 'plugin')) { return; }

            var ctx = TraceContextManager.getCurrentContext();
            if (ctx == null) { return; }

            if(funcName === 'open'){
                aop.functionHook(args, -1, function(obj, args){

                    var err = args[0];
                    var fd = args[1];
                    if (err) { return; }
                    if (openFiles[fd]) { return; }
                    openFiles[fd] = fileName;

                    //TODO: 파일정보 - 옵션처리 할 필요가 있음. fstat을 호출하면 트랜잭션 스택에 데이터가 수집됨.
                    // fs.fstat(fd, function(er, st) {
                    //     if (er || st && st.rdev !== 0) return; // not a regular file
                    // });
                });
            } else if(funcName === 'close') {
                var fd = fileName;
                if (!openFiles.hasOwnProperty(fd)) { return; }
                fileName = openFiles[fileName]; //fd를 파일명으로 변환
                delete openFiles[fd];
            }

            var msg = 'File',
                hash_middle = HashUtil.hashFromString(msg);

            DataTextAgent.MESSAGE.add(hash_middle, msg);

            var step = new MessageStep();
            step.hash = hash_middle;
            step.start_time = ctx.getElapsedTime();
            step.desc = funcName + ', fileName=\'' + fileName +'\'';
            ctx.profile.add(step);

            var cached_id = TraceContextManager.getCurrentId();
            aop.functionHook(args, -1, function (mod, args2) {

                if(cached_id == null) { return; }

                var ctx = TraceContextManager.resume(cached_id);
                if(ctx == null) { return; }
                cached_id = null;

                var laststep = ctx.profile.getLastSteps(1);
                if (laststep == null || laststep.length === 0) { return; }

                var err = args2[0];
                if(err) {
                    try {
                        var filePath = err.path, isRealFile = false;
                        var contextWithParam = filePath.substr(filePath.lastIndexOf('/') + 1);
                        if (contextWithParam.indexOf('.') > -1) {
                            if (contextWithParam.indexOf('?') > -1) {
                                contextWithParam = contextWithParam.substr(0, contextWithParam.indexOf('?'));
                            }
                            var fileEx = contextWithParam.substr(contextWithParam.lastIndexOf('.'));
                            for (var i = 0; i < extensions.length; i++) {
                                var ex = extensions[i];
                                if (ex === fileEx) {
                                    isRealFile = true;
                                    break;
                                }
                            }
                        }

                        if (isRealFile) {
                            if(ctx.service_name.endsWith('/')) { return; }
                            step.error = StatError.addError('file-'+err.code, err.message, 
                                ctx.service_hash);
                            if (ctx.error.isZero()) {
                                ctx.error = step.error;
                            }
                        }
                    } catch(e) {

                    }
                }
            });
        });
    });
};

exports.FileObserver = FileObserve;
exports.file = ingFile;
exports.fs = fileDescriptors;
