/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var TraceContextManager = require('../trace/trace-context-manager'),
    DataTextAgent       = require('../data/datatext-agent'),
    MessageStep         = require('../step/message-step'),
    HashUtil            = require('../util/hashutil'),
    DataTextAgent       = require('../data/datatext-agent'),
    URLPatternDetector  = require('../trace/serviceurl-pattern-detector').Detector,
    PathMap             = require('../trace/serviceurl-pattern-detector').PathMap,
    Logger              = require('../logger'),
    Long                = require('long');

var ExpressObserver = function(agent){
    this.agent = agent;
    this.packages = ['express'];
};

var methods = ['all', 'delete', 'get', 'head', 'opts', 'post', 'put', 'patch'];

var hookRestAPI = function(aop, mod){
    if (mod.__whatap_observe__) return;
    mod.__whatap_observe__ = true;

    methods.forEach(function (method) {
        if(mod[method]) aop.before(mod, method, function (obj, args) {
            aop.functionHook(args, -1, function (obj, args) {
                var ctx = TraceContextManager.getCurrentContext();
                if(ctx == null || args.length === 0) {return;}

                var req = args[0];
                if(Object.keys(req.params).filter(d=>d!=0).length > 0 && req.route && req.route.path && req.route.path != '*') {
                    ctx.service_name = ctx.service_name.replace(req.url, req.route.path);
                    ctx.service_hash = HashUtil.hashFromString(ctx.service_name);
                }
            })
       });
    });
}

var injectExpressV3 = function(aop, express){
    hookRestAPI(aop, express.Router.prototype);

    aop.before(express.Router, 'use', function(obj, args){
        wrapMiddlewareV3(aop, obj, args);
    });
}

var injectExpressV4 = function(aop, express){
    aop.after(express.Router, 'route', function(obj, args, create){
        hookRestAPI(aop, create);
        return create;
    })

    hookRestAPI(aop, express.Router);

    aop.before(express.Router, 'use', function(obj, args){
        wrapMiddlewareV4(aop, obj, args);
    });
}

ExpressObserver.prototype.inject = function( mod, moduleName ) {
    var self = this;
    if (mod.__whatap_observe__) return;
    mod.__whatap_observe__ = true;
    Logger.initPrint("ExpressObserver");
    var aop = this.agent.aop;
    
    if(!mod.Router){
        return;
    }else{
      if (mod.Router.use) {
        injectExpressV4(aop, mod)
      } else {
        injectExpressV3(aop, mod)
      }
    }

    aop.before(mod.Router.prototype, 'handle', function (obj, args) {
        if(args[0]) {
            TraceContextManager.resume(args[0].__ctx_id__);
        }
    });
};

function wrapMiddlewareV4(aop, obj, args) {
    var path = typeof args[0] === 'string' ? args[0] : '/',
        msg = 'Express',
        hash_middle = HashUtil.hashFromString(msg);

    if(typeof args[0] === 'string' && args[1]) {
        var tmp = args[1];
        if(tmp.stack) {
            try {
                for(var i=0; i < tmp.stack.length; i++) {
                    var routerPath = tmp.stack[i].route.path;
                    if(routerPath === '/') {
                        routerPath = '';
                    }
                    var url = args[0] + tmp.stack[i].route.routerPath;
                    if(url.indexOf(':') > 0) {
                        PathMap.pathParamUrlSet.add(url);
                    } else {
                        PathMap.noPathParamUrlHashSet.add(HashUtil.hashFromString(url));
                    }
                }
            } catch (e) {
            }
        }
    }
    URLPatternDetector.build(true);

    if(args.length > 1) {
        var func = args[args.length - 1];
        if(func.length > 3) {
            return;
        }
        var funcName = getFunctionName(func) || 'User defined function';
        aop.functionHook(args, -1, function (obj, args) {
            var ctx = TraceContextManager.getCurrentContext();
            if(ctx == null) { return; }

            var step = new MessageStep();
            step.hash = hash_middle;
            step.start_time = ctx.getElapsedTime();
            step.desc = funcName + ', path=\'' + path +'\' ';

            DataTextAgent.MESSAGE.add(hash_middle, msg);
            ctx.profile.add(step);

            aop.functionHook(args, -1, function (obj, args) {
                if(args[0] instanceof Error ) {
                    return;
                }
                step.time = ctx.getElapsedTime() - step.start_time;
            });
        });
    }
}

function wrapMiddlewareV3(aop, obj, args) {
    var path = typeof args[0] === 'string' ? args[0] : '/',
        msg = 'Express',
        hash_middle = HashUtil.hashFromString(msg);

    if(typeof args[0] === 'string' && args[1]) {
        var tmp = args[1];
        if(tmp.stack) {
            try {
                for(var i=0; i < tmp.stack.length; i++) {
                    var path = tmp.stack[i].route.path;
                    if(path === '/') {
                        path = '';
                    }
                    var url = args[0] + tmp.stack[i].route.path;
                    if(url.indexOf(':') > 0) {
                        PathMap.pathParamUrlSet.add(url);
                    } else {
                        PathMap.noPathParamUrlHashSet.add(HashUtil.hashFromString(url));
                    }
                }
            } catch (e) {
            }
        }
    }
    URLPatternDetector.build(true);

    if(args.length > 1) {
        var func = args[args.length - 1];
        if(func.length > 3) {
            return;
        }
        var funcName = getFunctionName(func) || 'User defined function';
        aop.functionHook(args, -1, function (obj, args) {
            var ctx = TraceContextManager.getCurrentContext();
            if(ctx == null) { return; }

            var step = new MessageStep();
            step.hash = hash_middle;
            step.start_time = ctx.getElapsedTime();
            step.desc = funcName + ', path=\'' + path +'\' ';

            DataTextAgent.MESSAGE.add(hash_middle, msg);
            ctx.profile.add(step);

            aop.functionHook(args, -1, function (obj, args) {
                if(args[0] instanceof Error ) {
                    return;
                }
                step.time = ctx.getElapsedTime() - step.start_time;
            });
        });
    }
}

function getFunctionName(func) {
    var ret = null;
    try {
        ret = func.toString();
        if(ret[0] === '(') {
            return null;
        }
        if(ret.indexOf('function ') >= 0 ) {
            ret = ret.substr('function '.length);
        }
        ret = ret.substr(0, ret.indexOf('('));
    } catch (e) {
    }
    return ret;
};

exports.ExpressObserver = ExpressObserver;