/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var net             = require('net'),
    netflag         = require('./netflag').NetFlag,
    netsrc          = require('./netflag').NetSrc,
    secu            = require('./security-master'),
    conf            = require('./../conf/configure'),
    DataOutputX     = require('./../io/data-outputx'),
    DataInputX      = require('./../io/data-inputx'),
    IPUtil          = require('./../util/iputil'),
    TcpReturn       = require('./tcp-return'),
    Receiver        = require('./receiver'),
    OidUtil         = require('../util/oidutil'),
    Logger          = require('../logger'),
    Long            = require('long');

function TcpSession() {
    if(typeof TcpSession.instance === 'object') {
        return TcpSession.instance;
    }

    this.in = null;
    this.out = null;
    this.receiver = new Receiver();
    this._cb = null;
    this.reconnectCnt = 0;
    TcpSession.instance = this;
}

TcpSession.prototype.open = function(cb){
    var self = this;
    
    if(cb) this._cb = cb;

    self.auto_oname_reset = conf.auto_oname_reset;
    if(secu.PCODE == null || secu.PCODE === 0) {
        if(cb) return cb(new Error('PCODE is not defined..'));
    }
    if(this.isOpen()) { 
        ++self.reconnectCnt;
        if(self.reconnectCnt >= 3) {
            self.close();
        }
        return cb(new Error('Already connected..')); 
    }

    var host = conf.getWhatapServerHost();
    var port = conf.whatap_server_port || 6600;
    var connection_timeout = conf.tcp_connection_timeout;

    var options = {
        host: host,
        port: port
    };

    Logger.print('WHATAP-180', "Try to connect to " + JSON.stringify(options), false);
    this.out = net.connect(options, function() {
        var localAddr = (this.localAddress || IPUtil.getIp());
        
        if(conf.auto_oname_enabled === true) {
            var oname = conf["whatap.oname"];
            if((oname == null || oname.length === 0)
                && self.auto_oname_reset === conf.auto_oname_reset) {
                self.prepareAutoOnameByServer(localAddr);
                return;
            }
        }
        
        secu.decideAgentOnameOid(localAddr);
        if(conf.cipher_net_enabled){
            try {
                self.out.write( self.keyReset() );
            } catch(e) {
                Logger.printError("WHATAP-125", "KeyReset " + host + ":" + port, e, true);
            }
        }
    });

    this.out.setKeepAlive(true , 60000);
    this.out.setTimeout(connection_timeout);

    this.out.on('error', function (err) {
        Logger.printError("WHATAP-127", 'Failed to connect to ' + JSON.stringify(options), err);
        self.close();
        cb(err);
    });

    this.out.on('timeout' , function () {
        Logger.print("WHATAP-184", "Socket timeout. WhaTap will retry connect after " + (connection_timeout/1000) + " sec.", false);
        self.close();
        setTimeout(function() {
            self.open(cb);
        } , 500);
    });

    var firstReceived = false;
    this.out.on('data', function(data) {
        if(self.PREPARE_AGENT) {
            var din = new DataInputX(data);
            var source = din.readByte();
            var code = din.readByte();
            var pcode = din.readLong();
            var oid = din.readInt();
            var transfer_key = din.readInt();
            var _data = din.readIntBytes(8 * 1024 * 1024);
            if (conf.cypher_level > 0) {
                _data = secu.cypher.decrypt(_data);
            }

            var oname = new DataInputX(_data).readText();
            
            var localAddr = (this.localAddress || IPUtil.getIp());
            secu.setAgentOnameOid(oname, localAddr);
            self.PREPARE_AGENT = false;
            if(conf.cipher_net_enabled) {
                try {
                    self.out.write( self.keyReset() );
                } catch(e) {
                    Logger.printError("WHATAP-125", "KeyReset " + host + ":" + port, e, true);
                }
            }
            return;
        }
        
        if(firstReceived) {
            self.read(data);
            return;
        }
        var d = self.readKeyReset(data);
        if( d && d.length != 0 ) {
            conf.updateNetCypherKey(d);
            firstReceived = true;
            if(cb) cb();
        }
    });

    this.out.on('end', function(){
        Logger.print("WHATAP-183", "Disconnected from server, please check your network or Whatap Server!!!!!", false);
        self.close();
        setTimeout(function() {
            cb(new Error('Socket end Event!!!!!'));
            return;
        },500);
    });
};

TcpSession.prototype.close = function(){
    if(this.out){ 
        this.out.destroy();
        this.out = null;
        this.reconnectCnt = 0;
    }
}

TcpSession.prototype.send = function(code , b ){
    try{
        if(this.out == null ){
            return false;
        }
        var out = new DataOutputX();
        out.writeByte(netsrc.AGENT_JAVA_EMBED);
        out.writeByte(code);
        out.writeLong(secu.PCODE);
        out.writeInt(secu.OID);
        out.writeInt(conf.TRANSFER_KEY);
        out.writeIntBytes(b);
        /*if(this.out === null ){
            if(this._cb) {
                setTimeout(function() {
                    this._cb(new Error('[Send Error] Socket is null !! TCP Session Reconnection......'));
                    return;
                },100);
         }
          } */
        this.out.write(out.toByteArray());
        return true;
    } catch(e) {
        Logger.printError('WHATAP-126', 'TcpSession Error', e);
    }
    this.close();
    return false;
};
TcpSession.prototype.readKeyReset = function( inPS /* DataInputX */){
    try {
        if( inPS.constructor === Buffer){
            inPS = new DataInputX(inPS);
        }
        var source = inPS.readByte();
        var code = inPS.readByte();
        var pcode = inPS.readLong();
        var oid = inPS.readInt();
        var transfer_key = inPS.readInt();
        var data = inPS.readIntBytes(1024);

        if( pcode != secu.PCODE || oid != secu.OID){
            return null;
        } else {
            return data;
        }
    } catch(e) {
        Logger.printError("WHATAP-189", 'License key or invalid agent version error(fail key)', e, true);
    }
};

TcpSession.prototype.keyReset = function(){
    var msg = new DataOutputX().writeText("hello").writeText(secu.ONAME).writeInt8(secu.IP).toByteArray();
    if( conf.cypher_level > 0 ){
        msg = secu.cypher.encrypt(msg);
    }

    var out = new DataOutputX();
    out.writeByte(netsrc.AGENT_JAVA_EMBED);
    out.writeByte(netflag.KEY_RESET);
    out.writeLong(Long.fromValue(secu.PCODE));
    out.writeInt(secu.OID);
    out.writeInt(0);
    out.writeIntBytes(msg);
    return out.toByteArray();
};

TcpSession.prototype.isOpen = function(){
    return this.out != null;
};

TcpSession.prototype.read = function (data) {
    var din = new DataInputX(data);
    var out = new TcpReturn();
    var source = din.readByte();

    out.code = din.readByte();
    var pcode = din.readLong();
    var oid = din.readInt();
    out.transfer_key = din.readInt();
    out.data = din.readIntBytes();

    if(pcode != secu.PCODE || oid != secu.OID) {
        Logger.printError('WHATAP-190', 'Received data is not mine..', new Error('Receive data error'));
        return null;
    }
    this.receiver.read(out);
};
TcpSession.prototype.prepareAutoOnameByServer = function (localAddr) {
    var self = this;
    try {
        var auto_oname_prefix = conf.auto_oname_prefix;

        if(auto_oname_prefix.indexOf('{') >= 0) {
            OidUtil.setIp(localAddr);
            OidUtil.setPort(port);
            OidUtil.setPid(process.pid);
            OidUtil.setHostName(require('os').hostname());
            auto_oname_prefix = OidUtil.mkOname(conf.auto_oname_prefix);
        }

        var pout = new DataOutputX();
        pout.writeText(auto_oname_prefix);
        pout.writeInt(IPUtil.stringToInt(localAddr));
        var msg = pout.toByteArray();

        var trkey = 0;
        if(conf.cypher_level > 0) {
            msg = secu.cypher.encrypt(msg);
            var b0 = 1;
            var b1 = (conf.cypher_level / 8);
            trkey = DataInputX.toInt([b0, b1, 0, 0], 0);
        }

        var out = new DataOutputX();
        out.writeByte(netsrc.AGENT_JAVA_EMBED);
        out.writeByte(netflag.PREPARE_AGENT);
        out.writeLong(secu.PCODE);
        out.writeInt(0);
        out.writeInt(trkey);
        out.writeIntBytes(msg);
        self.PREPARE_AGENT = true;
        self.out.write(out.toByteArray());
    } catch(e) {
        Logger.printError("WHATAP-003", "Prepare oname error", e, false);
    }
};

module.exports = new TcpSession();