/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var conf        = require('./../conf/configure'),
    DataOutputX = require('./../io/data-outputx'),
    EventPack   = require('./../pack/event-pack'),
    PackEnum    = require('./../pack/packenum'),
    EventLevel  = require('./../data/event-level'),
    NetFlag     = require('./netflag').NetFlag,
    TcpSession  = require('./tcp-session'),
    Logger      = require('../logger'),
    Long        = require('long');

var Sender = function(agent, requestMgr){
    this._agent = agent;
    this._mgr = requestMgr;
    this.next_time_sync = Date.now() + conf.getProperty('time_sync_interval_ms', 300000);
    this.isProcessing = false;
    this.ddd=Date.now()%100000;
};

Sender.prototype.startCheckQueue = function(){
    var self = this;
    if(this._interval) return;
    this.cypher_level = conf.cypher_level;
    this._interval = setInterval(function(){
        self.start();
    }, 100, 'whatap');
};
Sender.prototype.stopCheckQueue = function(){
    if(this._interval){
        clearInterval(this._interval);
        delete this._interval;
    }
};
Sender.prototype.start = function(){
    var self = this;
    this.isProcessing = true;
    var hasNext = self._mgr.pool.hasNext();

    if(!hasNext) {
        if(this._interval){
            return ;
        }else{
            this.startCheckQueue();
        }
    }
    this.stopCheckQueue();
    self.send();

};

Sender.prototype.send = function(){
    var self = this;
    var w = TcpSession;
    var p = self._mgr.pool.get();
    if(p == null) {return this.startCheckQueue();}
    if(w.out == null) {return this.startCheckQueue();}

    if(this.cypher_level != conf.cypher_level) {
        this.cypher_level = conf.cypher_level;
        w.close();
        return;
    }
    var len = 0;
    try{
        var now = Date.now();
        if( now > self.next_time_sync ){
            self.next_time_sync = now + conf.getProperty('time_sync_interval_ms', 300000);
            w.send(NetFlag.TIME_SYNC, DataOutputX.toBytesLong(Long.fromValue(now)));
        }
        if(conf.debug_tcpsend){
            try{
                var qsize=JSON.stringify(self._mgr.pool.size());
                var packName= PackEnum.PACK_NAME[p.pack.getPackType()]  ;
                var m = packName + "\t" +qsize;
                Logger.printAll("TCPSEND", m);
            }catch(e){
                Logger.printError('WHATAP-251', 'Sender(send)', e);
            }
        }

        p.pack.apType = 2;
        
        if(conf.cypher_level == 0){
            var b = DataOutputX.toBytesPack(p.pack);
            w.send(p.flag, b);
            len = b.length;
        } else {
            switch( NetFlag.getSecureMask(p.flag) ) {
                case NetFlag.SECURE_HIDE:
                    if(conf.cypher != null){
                        var b = DataOutputX.toBytesPack(p.pack);
                        b = conf.cypher.hide(b);
                        len = b.length;
                        w.send(p.flag, b);
                    }
                    break;
                case NetFlag.SECURE_CYPHER:
                    if(conf.cypher !== null){
                        var b = DataOutputX.toBytesPack(p.pack, conf.cypher_level / 8);
                        b = conf.cypher.encryptBlock(b);
                        len = b.length;
                        w.send(p.flag, b);
                    }
                    break;
                default:
                    var b = DataOutputX.toBytesPack(p.pack);
                    w.send(p.flag, b);
                    len = b.length;
                    break;
            }
        }

        if(len > conf.net_send_max_bytes){
            var e = new EventPack();
            e.level = EventLevel.FATAL;
            errorOverflow(p.pack.constructor + '('+ p.pack.getPackType() + ')');
            return;
        }

        self._mgr.sendBytes += len;
        self._mgr.sendPack++;
    } catch(e) {
        Logger.printError('WHATAP-121', 'Sender error', e);
    }

    process.nextTick(function(){
        self.start();
    });
};
var errorOverflow = function(packName){
    var e = new EventPack();
    e.title = "NEW_OVERFLOW";
    e.message = "Too big data: " + packName;
};
//PackEnum.PACK_NAME[p.pack.getPackType()]
module.exports = Sender;