/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var conf            = require('../../conf/configure'),
    DateUtil        = require('../../util/dateutil'),
    RequestQueue    = require('../../util/request-queue'),
    DataPackSender = require('../../data/datapack-sender'),
    DataOutputX = require('../../io/data-outputx'),
    ZipPack = require('../../pack/zip-pack'),
    Logger          = require('../../logger');
var zlib = require('zlib');

var logsink_enabled = conf.getProperty('logsink_enabled', false);
var logsink_zip_enabled = conf.getProperty('logsink_zip_enabled', false);
var logsink_max_buffer_size = conf.getProperty('logsink_max_buffer_size', 1024*64);
var profile_zip_min_size = conf.getProperty('profile_zip_min_size', 100)
var logsink_queue_size = conf.getProperty('logsink_queue_size', 500);
var logsink_max_wait_time = conf.getProperty('logsink_max_wait_time', 2000);
conf.on('logsink_enabled', function(newProperty) {
    logsink_enabled = newProperty;
    if (logsink_enabled && logsink_zip_enabled) {
        ZipSend.getInstance().startProcessQueue();
    } else {
        ZipSend.getInstance().stopProcessQueue();
        ZipSend.resetInstance();
    }
});
conf.on('logsink_zip_enabled', function(newProperty) {
    logsink_zip_enabled = newProperty;
    if (logsink_enabled && logsink_zip_enabled) {
        ZipSend.getInstance().startProcessQueue();
    } else {
        ZipSend.getInstance().stopProcessQueue();
        ZipSend.resetInstance();
    }
});

class ZipSend {
    constructor() {
        if (ZipSend.instance) {
            return ZipSend.instance;
        }
        this.queue = new RequestQueue(logsink_queue_size);
        this.buffer = Buffer.alloc(0);
        this.packCount = 0;
        this.first_time = null;
        this.isProcessing = false;
        ZipSend.instance = this;

        if (logsink_enabled && logsink_zip_enabled) {
            this.startProcessQueue();
        }
    }

    static getInstance() {
        if (!ZipSend.instance) {
            ZipSend.instance = new ZipSend();
        }
        return ZipSend.instance;
    }

    static resetInstance() {
        if(ZipSend.instance){
            ZipSend.instance = null;
        }
    }

    add(p){
        var ok = this.queue.put(p);
        if(!ok){
            DataPackSender.sendLogSinkPack(p);
        }
    }

    // async addWait(p, waitTimeForFull) {
    //     var ok = this.queue.put(p);
    //     if (ok === false){
    //         if(waitTimeForFull > 0){
    //             while(this.queue.put(p) === false){
    //                 await new Promise(resolve => setTimeout(resolve, waitTimeForFull));
    //             }
    //         }
    //     }
    // }

    async processQueue() {
        this.isProcessing = true;
        while (this.isProcessing) {
            if(!logsink_enabled || !logsink_zip_enabled) {
                this.isProcessing = false;
                break;
            }

            const p = await this.queue.getByTimeout(logsink_max_wait_time);
            if(p) {
                await this.append(p);
            }else{
                await this.sendAndClear();
            }
            await new Promise(resolve => setTimeout(resolve, 100));
        }
    }

    async append(p) {
        var b = DataOutputX.toBytesPack(p);
        this.packCount++;
        this.buffer = Buffer.concat([this.buffer, b]);

        if(!this.first_time){
            this.first_time = p.time;
            if (this.buffer.length >= logsink_max_buffer_size) {
                await this.sendAndClear();
            }
        }else{
            if(this.buffer.length >= logsink_max_buffer_size || (p.time - this.first_time) >= conf.getProperty('logsink_max_wait_time', 2000)){
                await this.sendAndClear();
            }
        }
    }

    async sendAndClear() {
        if(this.buffer.length === 0){ return; }

        var p = new ZipPack();
        p.time = DateUtil.currentTime();
        p.recordCount = this.packCount;
        p.records = this.buffer;

        await this.doZip(p);
        if(conf.getProperty('debug_logsink_zip_enabled', false)){
            Logger.print('WHATAP-703', `zip status=${p.status} records=${p.recordCount} | ${this.buffer.length} => ${p.records.length}`);
        }
        DataPackSender.sendLogSinkPack(p);

        this.buffer = Buffer.alloc(0);
        this.first_time = 0;
        this.packCount = 0;
    }

    async doZip(p){
        if(p.status !== 0){ return; }
        if(p.records.length < profile_zip_min_size){ return; }
        p.status = 1;

        try{
            p.records = await this._doZip(p.records);
        }catch (error) {
            Logger.print('WHATAP-701', 'Error occurred during compression.' + error, false);
        }
    }

    _doZip(data) {
        return new Promise((resolve, reject) => {
            zlib.gzip(data, (err, buffer) => {
                if (err) {
                    reject(err);
                } else {
                    resolve(buffer);
                }
            });
        });
    }

    startProcessQueue() {
        if (!this.isProcessing) {
            this.processQueue()
        }
    }
    stopProcessQueue() {
        this.isProcessing = false;
    }
}

module.exports = ZipSend;