const conf = require('../conf/configure');
const EventPack = require('../pack/event-pack');
const DataPackSender = require('../data/datapack-sender');
const Logger = require('../logger');
const EventLevel = require('../data/event-level');

class LineLogUtil {
    constructor() {
        this.last_log = Date.now();
        this.last_alert = Date.now();
    }

    static getInstance() {
        if (!LineLogUtil.instance) {
            LineLogUtil.instance = new LineLogUtil();
        }
        return LineLogUtil.instance;
    }

    checkLogContent(lineLog, orgContent) {
        if (!orgContent) {
            return;
        }

        const len = orgContent.length;
        let maxLength = conf.getProperty('logsink_limit_content_length', 10000);
        if(maxLength > 10000){
            maxLength = 10000;
        }

        if ((conf.getProperty('logsink_limit_content_enabled', true) && len > maxLength) || len > 10000) {
            lineLog.content = orgContent.substring(0, maxLength) + " ...(truncated)";
            lineLog.truncated = true;

            this.addTruncateTag(lineLog);

            const alertMessage = lineLog.category + " truncated (len=" + len + ")";
            this.logAndAlert(alertMessage);
        } else {
            lineLog.content = orgContent;
        }
    }

    log(message) {
        if (!conf.getProperty('logsink_max_content_log_enabled', true)) {
            return;
        }

        const now = Date.now();
        if (now - this.last_log < conf.getProperty('logsink_max_content_log_silent_time', 60000)) {
            return;
        }
        this.last_log = now;
        Logger.print('WHATAP-101', message, false);
    }

    alertOverflow(message) {
        if (!conf.getProperty('logsink_max_content_alert_enabled', false)) {
            return;
        }

        const now = Date.now();
        if (now - this.last_alert < conf.getProperty('logsink_max_content_alert_silent_time', 60000)) {
            return;
        }
        this.last_alert = now;

        const eventPack = new EventPack();
        eventPack.level = EventLevel.FATAL;
        eventPack.title = "LOG_TRUNCATED";
        eventPack.message = "Too big Log: " + message;
        DataPackSender.sendEventPack(eventPack);
    }

    logAndAlert(alertMessage) {
        this.log(alertMessage);
        this.alertOverflow(alertMessage);
    }

    addTruncateTag(lineLog) {
        lineLog.tags.putString("truncated", "true");
    }
}

LineLogUtil.instance = null;

module.exports = LineLogUtil;