/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var constants   = require('../env/constants'),
    util        = require('../util/utils'),
    Long        = require('long'),
    Logger      = require('../logger');

var BYTE1_MAX_UVALUE = 0xff,
    BYTE2_MAX_UVALUE = 0xffff,
    BYTE4_MAX_UVALUE = 0xffffffff,
    BYTE8_MAX_UVALUE = 0xffffffffffffffff,
    BYTE1_MAX_VALUE = Math.floor( BYTE1_MAX_UVALUE / 2 ),
    BYTE1_MIN_VALUE = Math.floor( -BYTE1_MAX_UVALUE / 2 ),
    BYTE2_MAX_VALUE = Math.floor( BYTE2_MAX_UVALUE / 2 ),
    BYTE2_MIN_VALUE = Math.floor( -BYTE2_MAX_UVALUE / 2 ),
    BYTE4_MAX_VALUE = Math.floor( BYTE4_MAX_UVALUE / 2 ),
    BYTE4_MIN_VALUE = Math.floor( -BYTE4_MAX_UVALUE / 2 ),
    BYTE8_MAX_VALUE = Math.floor( BYTE8_MAX_UVALUE / 2 ),
    BYTE8_MIN_VALUE = Math.floor( -BYTE8_MAX_UVALUE / 2 );

function isNumOk(n) {
    return n != null && n !== undefined && isNaN(n) == false;
}

function isNotNull(n) {
    return n != null && n !== undefined;
}

function DataOutputX(size) {
    this.buf = undefined;
    if (size == undefined) {
        this.buf = Buffer.alloc(32 , 0);
    } else {
        this.buf = Buffer.alloc(size , 0);
    }
    this.offset = 0;
}

DataOutputX.prototype.ensureCapacity = function(minCapacity) {
    if (minCapacity - this.buf.length > 0) {
        this.grow(minCapacity);
    }
};

DataOutputX.prototype.grow = function(minCapacity) {
    var oldCapacity = this.buf.length;
    var newCapacity = oldCapacity << 1;
    if (newCapacity - minCapacity < 0) {
        newCapacity = minCapacity;
    }
    if (newCapacity < 0) {
        if (minCapacity < 0) {
            throw 'out of memory';
        }
        newCapacity = 2147483647;
    }
    var oldBuffer = this.buf;
    this.buf = Buffer.alloc(newCapacity,0);
    oldBuffer.copy(this.buf);

};

DataOutputX.prototype.reset = function() {
    this.offset = 0;
};

DataOutputX.prototype.size = function() {
    return this.buf.length;
};

DataOutputX.prototype.writeIntBytes = function(b){
    if( !b || b.length == 0 ){
        this.writeInt(0);
    } else{
        this.writeInt(b.length);
        this.writeBytes(b);
    }
};

DataOutputX.prototype.writeShortBytes = function(b) {
    this.writeShort(b.length);
    this.writeBytes(b);
    return this;
};

DataOutputX.prototype.writeBlob = function(value) {
    if (isNotNull(value) == false) {
        this.writeByte(0);
    } else {
        if((value instanceof Buffer) == false) {
            throw new Error('Write Blob parameter must be buffer instance');
        }
        var len = value.length;
        if (len <= 253) {
            this.writeUInt8(len);
            this.copy(value);
        } else if (len <= 65535) {
            this.writeUInt8(255);
            this.writeUInt16BE(len);
            this.copy(value);
        } else {
            this.writeUInt8(254);
            this.writeUInt32BE(len);
            this.copy(value);
        }
    }
    return this;
};

DataOutputX.prototype.copy = function(inputBuff) {
    this.ensureCapacity(this.offset + inputBuff.length);
    inputBuff.copy(this.buf, this.offset, 0, inputBuff.length);
    this.offset += inputBuff.length;
    inputBuff = null;

};

DataOutputX.prototype.toByteArray = function() {
    var returnBuffer = Buffer.alloc(this.offset);
    this.buf.copy(returnBuffer);
    return returnBuffer;
};

DataOutputX.prototype.writeByte = function(b) {
    this.ensureCapacity(this.offset + 1);
    this.buf[this.offset] = b;
    this.offset += 1;
    return this;
};

DataOutputX.prototype.writeBytes = function(bytes) {
    if (isNotNull(bytes) == false) {
        bytes = [];
    }

    this.ensureCapacity(this.offset + bytes.length);
    for ( var i = 0; i < bytes.length; i++) {
        this.buf[this.offset] = bytes[i];
        this.offset += 1;
    }

    return this;
};

DataOutputX.prototype.writeDecimal = function(value) {
    if(value instanceof Long) {
        this.writeByte(8);
        this.writeLong(value);
        return this;
    }

    if (isNumOk(value) == false) {
        this.writeByte(0);
    } else if (constants.numType.INT8_MIN <= value
        && value <= constants.numType.INT8_MAX) {
        this.writeByte(1);
        this.writeInt8(value);
    } else if (constants.numType.INT16_MIN <= value
        && value <= constants.numType.INT16_MAX) {
        this.writeByte(2);
        this.writeInt16BE(value);
    } else if (constants.numType.INT24_MIN <= value
        && value <= constants.numType.INT24_MAX) {
        this.writeByte(3);
        this.writeInt24BE(value);
    } else if (constants.numType.INT32_MIN <= value
        && value <= constants.numType.INT32_MAX) {
        this.writeByte(4);
        this.writeInt32BE(value);
    } else if(constants.numType.INT40_MIN <= value && value <=
        constants.numType.INT40_MAX) {
        this.writeByte(5);
        this.writeInt40BE(value);
    } else if (constants.numType.INT64_MIN <= value
        && value <= constants.numType.INT64_MAX) {
        var longVal = Long.fromNumber(value);
        this.writeByte(8);
        this.writeLong(longVal);
    }
    return this;
};

DataOutputX.prototype.write = function(value /* byte[] */, off /* int */, len /* int */){
    this.writeBytes(value);
};

DataOutputX.prototype.writeInt = function(value){
    this.writeInt32BE(value);
};

DataOutputX.prototype.writeInt8 = function(value) {
    if (isNumOk(value) == false) {
        value = 0;
    }
    this.ensureCapacity(this.offset + 1);
    this.buf[this.offset] = value;
    this.offset += 1;
    return this;
};

DataOutputX.prototype.writeUInt8 = function(value) {
    if (isNumOk(value) == false) {
        value = 0;
    }
    this.ensureCapacity(this.offset + 1);
    // this.buf[this.offset] = value;
    this.buf.writeUInt8(value,this.offset);
    this.offset += 1;
    return this;
};

DataOutputX.prototype.writeShort = function(value) {
    this.writeInt16BE(value);
};

DataOutputX.prototype.writeInt16BE = function(value) {
    if (isNumOk(value) == false) {
        value = 0;
    }
    this.ensureCapacity(this.offset + 2);
    this.buf.writeInt16BE(value, this.offset);
    this.offset += 2;
    return this;
};

DataOutputX.prototype.writeUInt16BE = function(value) {
    if (isNumOk(value) == false) {
        value = 0;
    }
    this.ensureCapacity(this.offset + 2);
    this.buf.writeUInt16BE(value, this.offset);
    this.offset += 2;
    return this;
};

DataOutputX.prototype.writeInt24BE = function(value) {
    if (isNumOk(value) == false) {
        value = 0;
    }
    this.ensureCapacity(this.offset + 1);
    this.buf[this.offset] = (value >>> 16) & 0xff;
    this.buf[this.offset + 1] = (value >>> 8) & 0xff;
    this.buf[this.offset + 2] = (value >>> 0) & 0xff;
    this.offset += 3;
    return this;
};

DataOutputX.prototype.writeInt32BE = function(value) {
    if (isNumOk(value) == false) {
        value = 0;
    }
    this.ensureCapacity(this.offset + 4);
    this.buf.writeInt32BE(value, this.offset);
    this.offset += 4;
    return this;
};

DataOutputX.prototype.writeUInt32BE = function(value) {
    if (isNumOk(value) == false) {
        value = 0;
    }
    this.ensureCapacity(this.offset + 4);
    this.buf.writeUInt32BE(value, this.offset);
    this.offset += 4;
    return this;
};

DataOutputX.prototype.writeInt40BE = function(value) {
    if (isNumOk(value) == false) {
        value = 0;
    }
    var lo = value % 0x100000000;
    var hi = value / 0x100000000;
    hi = hi | 0;

    this.writeByte((hi >>> 0) & 0xff);
    this.writeByte((lo >>> 24) & 0xff);
    this.writeByte((lo >>> 16) & 0xff);
    this.writeByte((lo >>> 8) & 0xff);
    this.writeByte((lo >>> 0) & 0xff);

    return this;
};

DataOutputX.prototype.writeLong = function(value){
    var v = value;
    if(typeof value == 'string'){
        v = Long.fromString(value);
    } else if(typeof value == 'number') {
        v = Long.fromNumber(value);
    }

    if(v.constructor == Long){
        this.writeInt32BE(v.getHighBits())
        this.writeInt32BE(v.getLowBits())
    }else{
        this.ensureCapacity(this.offset + 8);
        this.offset += 8;
    }
    return this;
};

DataOutputX.prototype.writeInt64BE = function(value) {
    if (isNumOk(value) == false) {
        value = 0;
    }
    var lo = value % 0x10000000;
    var hi = value / 0x10000000;
    hi = hi | 0;

    this.writeByte((hi >>> 24) & 0xff);
    this.writeByte((hi >>> 16) & 0xff);
    this.writeByte((hi >>> 8) & 0xff);
    this.writeByte((hi >>> 0) & 0xff);
    this.writeByte((lo >>> 24) & 0xff);
    this.writeByte((lo >>> 16) & 0xff);
    this.writeByte((lo >>> 8) & 0xff);
    this.writeByte((lo >>> 0) & 0xff);

    return this;
};

DataOutputX.prototype.writeTXID = function(int64) {
    int64.copy(this.buf, this.offset);
    this.offset += 8;
    return this;
};

DataOutputX.prototype.writeFloat = function(value) {
    if (isNumOk(value) == false) {
        value = 0.0;
    }
    this.ensureCapacity(this.offset + 4);
    this.buf.writeFloatBE(value, this.offset);
    this.offset += 4;
    return this;
};

DataOutputX.prototype.writeDouble = function(value) {
    if (isNumOk(value) == false) {
        value = 0.0;
    }
    this.ensureCapacity(this.offset + 8);
    this.buf.writeDoubleBE(value, this.offset);
    this.offset += 8;
    return this;
};

DataOutputX.prototype.writeString = function(value) {
    var length = value.length;
    this.ensureCapacity(this.offset + length);
    this.buf.write(value, this.offset, length);
    this.offset += length;
    return this;

};

DataOutputX.prototype.writeText = function(value) {
    if (isNotNull(value) == false) {
        this.writeByte(0);
    } else {

        if(typeof value === 'string') {
            value = util.toUTF8(value);
        }
        if(Array.isArray(value)) {
            var len = value.length;
            if (len <= 253) {
                this.writeUInt8(len);
                this.writeBytes(value);
            } else if (len <= 65535) {
                this.writeUInt8(255);
                this.writeUInt16BE(len);
                this.writeBytes(value);
            } else {
                this.writeUInt8(254);
                this.writeUInt32BE(len);
                this.writeBytes(value);
            }
        } else {
            this.writeByte(0);
        }
    }
    return this;
};

DataOutputX.prototype.print = function() {
    for ( var i = 0; i < this.offset; i++) {
        console.log(i + ': ' + this.buf[i].toString(16));
    }
};

DataOutputX.prototype.writePack = function(pack) {
    this.writeInt16BE(pack.getPackType());
    pack.write(this);
    return this;
};

DataOutputX.prototype.writeValue = function(value) {
    if(value == null){
        // value = NullValue
    }
    try{
        this.writeByte(value.getValueType());
        value.write(this);
    } catch (e){
        Logger.printError('WHATAP-251', 'DataOutputX writeValue Error', e);
    }

    return this;
};

DataOutputX.prototype.writeStep = function(step) {
    this.writeByte(step.getStepType());
    step.write(this);
    return this;
};

DataOutputX.prototype.writeBoolean = function(value) {
    if(typeof value == 'boolean') {
        this.writeByte(value ? 1 : 0);
        return this;
    } else {
        this.writeByte(0);
        return this;
    }
};

DataOutputX.prototype.writeFloatArray = function(value) {
    if(value == null || Array.isArray(value) == false) {
        this.writeShort(0);
    } else {
        var length = value.length;
        this.writeShort(length);
        for(var i=0; i<length; i++) {
            this.writeFloat(value[i]);
        }
    }
    return this;
};

DataOutputX.prototype.writeIntArray = function(value) {
    if(value == null || Array.isArray(value) == false) {
        this.writeShort(0);
    } else {
        var length = value.length;
        this.writeShort(length);
        for(var i=0; i<length; i++) {
            this.writeInt(value[i]);
        }
    }
    return this;
};

DataOutputX.prototype.writeLongArray = function(value) {
    if(value == null || Array.isArray(value) == false) {
        this.writeShort(0);
    } else {
        var length = value.length;
        this.writeShort(length);
        for(var i=0; i<length; i++) {
            var long = value[i];
            if(long instanceof Long) {
                this.writeLong(value[i]);
            } else {
                this.writeLong(Long.fromInt(0));
            }
        }
    }
};

DataOutputX.toBytes = function(v) {
    var buf = new Array(4);
    buf[0] = ((v >>> 24) & 0xFF);
    buf[1] = ((v >>> 16) & 0xFF);
    buf[2] = ((v >>> 8) & 0xFF);
    buf[3] = ((v >>> 0) & 0xFF);
    return buf;
};

DataOutputX.toBytesLong = function(value) {
    var v = value;
    if(v.constructor == String){
        v = Long.fromString(value);
    }

    var tmp = Buffer.alloc(8);
    if(v.constructor == Long){
        tmp.writeInt32BE(v.getHighBits(), 0);
        tmp.writeInt32BE(v.getLowBits(), 4);
    }else{
        tmp.writeInt32BE(0);
        tmp.writeInt32BE(0);
    }
    var high = tmp.readInt32BE(0);
    var low = tmp.readInt32BE(4);
    return tmp;
};

// DataOutputX.toByteArray = function(v, byteLength){
//     var buf;
//     if( byteLength == 2 && (BYTE2_MIN_VALUE <= v && v <= BYTE2_MAX_VALUE) ){
//         byteLength = byteLength || 2;
//         buf = new Buffer(byteLength);
//         buf[0] = ((v >>> 8) & 0xFF);
//         buf[1] = ((v >>> 0) & 0xFF);
//     }else if( byteLength == 4 && (BYTE4_MIN_VALUE <= v && v <= BYTE4_MAX_VALUE) ){
//         byteLength = byteLength || 4;
//         buf = new Buffer(byteLength);
//         buf[0] = ((v >>> 24) & 0xFF);
//         buf[1] = ((v >>> 16) & 0xFF);
//         buf[2] = ((v >>> 8) & 0xFF);
//         buf[3] = ((v >>> 0) & 0xFF);
//     }else if( byteLength == 8 && (BYTE8_MIN_VALUE <= v && v <= BYTE8_MAX_VALUE) ){
//         byteLength = byteLength || 8;
//         buf = new Buffer(byteLength);
//         buf[0] = (v >>> 56);
//         buf[1] = (v >>> 48);
//         buf[2] = (v >>> 40);
//         buf[3] = (v >>> 32);
//         buf[4] = (v >>> 24);
//         buf[5] = (v >>> 16);
//         buf[6] = (v >>> 8);
//         buf[7] = (v >>> 0);
//     }
//     return buf;
// };

DataOutputX.set = function(dest, pos, src){
    src.copy(dest, 0, pos, src.length);
    return dest;
};

DataOutputX.toBytesPack = function(pack, fmtLen) {
    if(fmtLen == undefined ) {
        return new DataOutputX().writePack(pack).toByteArray();
    }
    var out = new DataOutputX().writePack(pack);
    var remainder = out.size() % (fmtLen||0);
    if (remainder != 0) {
        out.write(Buffer.alloc(fmtLen - remainder));
    }
    return out.toByteArray();
}

module.exports = DataOutputX;