/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var conf            = require('../conf/configure'),
    secuMaster      = require('../net/security-master'),
    TcpRequestMgr   = require('../net/tcprequest-mgr'),
    DateUtil        = require('./../util/dateutil'),
    RequestQueue    = require('../util/request-queue'),
    DataPackSender = require('../data/datapack-sender'),
    DataOutputX = require('../io/data-outputx'),
    ZipPack = require('../pack/zip-pack'),
    DataPackSender = require('../data/datapack-sender'),
    Logger          = require('../logger');

const zlib = require('zlib');

var profile_zip_enabled = conf.getProperty('profile_zip_enabled', false);
var profile_zip_max_buffer_size = conf.getProperty('profile_zip_max_buffer_size', 1024*1024);
var profile_zip_min_size = conf.getProperty('profile_zip_min_size', 100)
var profile_zip_queue_size = conf.getProperty('profile_zip_queue_size', 500);
var profile_zip_max_wait_time = conf.getProperty('profile_zip_max_wait_time', 1000);
var net_send_max_bytes = conf.getProperty('net_send_max_bytes', 5 * 1024 * 1024);
var debug_profile_zip_enabled = conf.getProperty('debug_profile_zip_enabled', false);
var debug_profile_zip_interval = conf.getProperty('debug_profile_zip_interval', 5000);
conf.on('profile_zip_enabled', function(newProperty) {
    profile_zip_enabled = newProperty;
    if (profile_zip_enabled) {
        ZipProfile.getInstance().startProcessQueue();
    } else {
        ZipProfile.getInstance().stopProcessQueue();
        ZipProfile.resetInstance();
    }
});

class ZipProfile {
    constructor() {
        if (ZipProfile.instance) {
            return ZipProfile.instance;
        }
        this.queue = new RequestQueue(profile_zip_queue_size);
        this.no_zip_sent = 0;
        this.zip_sent = 0;
        this.last_log = null;
        this.buffer = Buffer.alloc(0);
        this.packCount = 0;
        this.first_time = null;
        this.isProcessing = false;
        ZipProfile.instance = this;

        if (profile_zip_enabled) {
            this.startProcessQueue();
        }
    }

    static getInstance() {
        if (!ZipProfile.instance) {
            ZipProfile.instance = new ZipProfile();
        }
        return ZipProfile.instance;
    }

    static resetInstance() {
        if(ZipProfile.instance){
            ZipProfile.instance = null;
        }
    }

    add(p){
        var ok = this.queue.put(p);
        if(!ok){
            DataPackSender.sendPack(p);
            this.no_zip_sent++;
        }
    }

    async processQueue() {
        this.isProcessing = true;
        while (this.isProcessing) {
            if(!profile_zip_enabled) { break; }

            const p = await this.queue.getByTimeout(profile_zip_max_wait_time);
            if(p) {
                await this.append(p);
            }else{
                await this.sendAndClear();
            }
            await new Promise(resolve => setTimeout(resolve, 100));
        }
    }

    async append(p) {
        var b = DataOutputX.toBytesPack(p);
        if(b.length >= net_send_max_bytes){ return; }

        this.buffer = Buffer.concat([this.buffer, b]);
        this.packCount++;

        if(!this.first_time){
            this.first_time = p.time;
            if(this.buffer.length >= profile_zip_max_buffer_size){
                await this.sendAndClear();
            }
        }else{
            if(this.buffer.length >= profile_zip_max_buffer_size || (p.time - this.first_time) >= conf.getProperty('profile_zip_max_wait_time', 5000)){
                await this.sendAndClear();
            }
        }
    }

    async sendAndClear() {
        if(this.buffer.length === 0){ return; }

        var p = new ZipPack();
        p.time = DateUtil.currentTime();
        p.recordCount = this.packCount;
        p.records = this.buffer;

        await this.doZip(p);
        if(debug_profile_zip_enabled){
            if(debug_profile_zip_interval){
                Logger.print('WHATAP-ZIP-DEBUG', `PROFILE status=${p.status} records=${p.recordCount} | ${this.buffer.length} => ${p.records.length} queue=${this.queue.size()}`, false)
            } else {
                this.zip_sent++;
                var now = Date.now();
                if(now > (this.last_log + debug_profile_zip_interval)){
                    this.last_log = now;
                    this.log(p);
                    this.zip_sent = 0;
                }
            }
        }

        p.pcode = secuMaster.PCODE;
        p.oid = secuMaster.OID;
        p.okind = conf.OKIND;
        p.onode = conf.ONODE;

        TcpRequestMgr.addProfile(0, p);

        this.buffer = Buffer.alloc(0);
        this.first_time = 0;
        this.packCount = 0;
    }

    log(p){
        try{
            var sb = `PROFILE `;
            sb += `zip_sent=${this.zip_sent}`;
            sb += ` records=${p.records}`;
            sb += ` | =${this.buffer.length} => ${p.records.length}`;
            sb += ` queue=${this.queue.size()}`;
            if (this.no_zip_sent > 0) {
                sb += ` no_zip_sent=${this.no_zip_sent}`;
            }
            Logger.print('WHATAP-ZIP-DEBUG', sb, false);
        }catch (e) {
        }
    }

    async doZip(p){
        if(p.status !== 0){ return; }
        if(p.records.length < profile_zip_min_size){ return; }
        p.status = 1;

        try{
            p.records = await this._doZip(p.records);
        }catch (e) {
            Logger.print('WHATAP-ZIP-ERROR', 'Error occurred during compression.', false);
        }
    }

    _doZip(data) {
        return new Promise((resolve, reject) => {
            zlib.gzip(data, (err, buffer) => {
                if (err) {
                    reject(err);
                } else {
                    resolve(buffer);
                }
            });
        });
    }

    startProcessQueue() {
        if (!this.isProcessing) {
            this.processQueue()
        }
    }
    stopProcessQueue() {
        this.isProcessing = false;
    }
}

module.exports = ZipProfile;