/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */
var RequestQueue    = require('../util/request-queue'),
    TextPack        = require('../pack/text-pack').TextPack,
    TextRec         = require('../pack/text-pack').TextRec,
    TextTypes       = require('../lang/text-types'),
    DataPackSender  = require('../data/datapack-sender'),
    DateUtil        = require('../util/dateutil'),
    IntSet        = require('../util/intset');

const BUFFERD_MAX   = 10 * 1024;
const INTERVAL      = 500;   // 제어 가능해야 한다.

const DBC               = new HashedData(TextTypes.DB_URL, 1000);
const METHOD            = new HashedData(TextTypes.METHOD, 5000);
const HTTPC_URL         = new HashedData(TextTypes.HTTPC_URL, 2000);
const HTTPC_HOST        = new HashedData(TextTypes.HTTPC_HOST, 2000);
const ERROR             = new HashedData(TextTypes.ERROR, 1000);
const SERVICE           = new HashedData(TextTypes.SERVICE, 2000);
const SQL               = new HashedData(TextTypes.SQL, 2000);
const STACK             = new HashedData(TextTypes.STACK_ELEMENTS, 20000);
const MESSAGE           = new HashedData(TextTypes.MESSAGE, 2000);
const USERAGENT         = new HashedData(TextTypes.USER_AGENT, 1000);
const USER_AGENT_OS     = new HashedData(TextTypes.USER_AGENT_OS, 1000);
const USER_AGENT_BROWSER= new HashedData(TextTypes.USER_AGENT_BROWSER, 1000);
const REFERER           = new HashedData(TextTypes.REFERER, 1000);
const HTTP_DOMAIN       = new HashedData(TextTypes.HTTP_DOMAIN, 500);
const MTRACE_SPEC       = new HashedData(TextTypes.MTRACE_SPEC, 500);
const MTRACE_CALLER_URL       = new HashedData(TextTypes.MTRACE_CALLER_URL, 500);
const OKIND       = new HashedData(TextTypes.OKIND_NAME, 100);
const ONODE       = new HashedData(TextTypes.ONODE_NAME, 100);

var DataTextAgent = function () {

    if(typeof DataTextAgent.instance == 'object') {
        return DataTextAgent.instance;
    }

    this.date = DateUtil.yyyymmddhh();
    this.queue = new RequestQueue(500);
    this.bufferPack = new TextPack();
    this.bufferedLength = 0;
    this.run();

    DataTextAgent.instance = this;
};

DataTextAgent.prototype.run = function () {
    var r = this.queue.get();
    if(r !== null) {
        this.bufferPack.records.push(r);
        this.bufferedLength += r.text.length;
        if(this.bufferedLength >= BUFFERD_MAX) {
            this.send();
        }
    } else {
        if(this.bufferedLength > 0) {
            this.send();
        }
    }

    if(this.date !== DateUtil.yyyymmddhh()) {
        this.reset();
        this.date = DateUtil.yyyymmddhh();
    }

    var self = this;
    setTimeout(function () {
        self.run();
    }, INTERVAL, 'whatap');
};
DataTextAgent.prototype.add = function (type, hash, text) {
    this.queue.put(new TextRec(type, hash, text));
};
DataTextAgent.prototype.send = function () {
    DataPackSender.sendTextPack(this.bufferPack);
    this.bufferPack = new TextPack();
    this.bufferedLength = 0;
};
DataTextAgent.prototype.reset = function () {
    DBC.sent.clear();
    METHOD.sent.clear();
    HTTPC_URL.sent.clear();
    HTTPC_HOST.sent.clear();
    ERROR.sent.clear();
    SERVICE.sent.clear();
    SQL.sent.clear();
    STACK.sent.clear();
    MESSAGE.sent.clear();
    USERAGENT.sent.clear();
    USER_AGENT_OS.sent.clear();
    USER_AGENT_BROWSER.sent.clear();
    REFERER.sent.clear();
    HTTP_DOMAIN.sent.clear();
    MTRACE_SPEC.sent.clear();
    MTRACE_CALLER_URL.sent.clear();
    OKIND.sent.clear();
    ONODE.sent.clear();
};

function HashedData(type, n) {
    this.type = type || 0;
    this.sent = new IntSet().setMax(n);
}
HashedData.prototype.add = function (hash, text) {
    if(require('../trace/trace-context-manager').initialized === false) { return; }
    // 보내지 않은 Hash인 경우 추가
    if(this.sent.hasKey(hash) == false) {
        this.sent.add(hash);
        new DataTextAgent().add(this.type, hash, text);
    }
};

exports.agent = new DataTextAgent();
exports.DBC = DBC;
exports.METHOD = METHOD;
exports.HTTPC_URL = HTTPC_URL;
exports.HTTPC_HOST = HTTPC_HOST;
exports.ERROR = ERROR;
exports.SERVICE = SERVICE;
exports.SQL = SQL;
exports.STACK = STACK;
exports.MESSAGE = MESSAGE;
exports.USERAGENT = USERAGENT;
exports.USER_AGENT_OS = USER_AGENT_OS;
exports.USER_AGENT_BROWSER = USER_AGENT_BROWSER;
exports.REFERER = REFERER;
exports.HTTP_DOMAIN = HTTP_DOMAIN;
exports.MTRACE_SPEC = MTRACE_SPEC;
exports.MTRACE_CALLER_URL = MTRACE_CALLER_URL;
exports.OKIND = OKIND;
exports.ONODE = ONODE;
