/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var DataPackSender  = require('./datapack-sender'),
    StatTranX       = require('../stat/stat-tranx'),
    StatRemoteIp    = require('../stat/stat-remote-ip'),
    StatUserAgent   = require('../stat/stat-useragent'),
    StatTranxDomain   = require('../stat/stat-tx-domain'),
    StatTranxMtCaller   = require('../stat/stat-tx-caller'),
    StatTranxReferer   = require('../stat/stat-tx-referer'),
    StatRemoteIpUrl   = require('../stat/stat-remote-ipurl'),
    conf            = require('../conf/configure'),
    HashUtil            = require('../util/hashutil'),
    EventLevel          = require('../data/event-level'),
    DataTextAgent       = require('../data/datatext-agent'),
    Long = require('long'),
    Logger          = require('../logger');

var ZipProfile = require('./zipprofile');
var profile_zip_enabled = conf.getProperty('profile_zip_enabled', false);
var stat_ipurl_enabled = conf.getProperty('stat_ipurl_enabled', false);
conf.on('profile_zip_enabled', function(newProperty) {
    profile_zip_enabled = newProperty;
});
conf.on('stat_ipurl_enabled', function(newProperty) {
    stat_ipurl_enabled = newProperty;
});

var DataProfileAgent = {
    last_reject : 0,
    sendProfile : function (ctx, profile, rejected) {
        try {
            StatRemoteIp.incRemoteIp(ctx.remoteIp);
            StatUserAgent.incUserAgent(ctx.userAgent);

            var transaction = profile.service;
            if(stat_ipurl_enabled && ctx.remoteIp){
                var tc = StatRemoteIpUrl.getService(ctx.remoteIp, ctx.service_hash);
                if (tc) {
                    tc.count++;
                    if (tc.errorLevel >= EventLevel.WARNING) {
                        tc.error++;
                    }
                    tc.time += transaction.elapsed;
                }
            }

            if(conf.stat_domain_enabled && ctx.http_host_hash!=0) {
                StatTranxDomain.add(ctx.http_host_hash, ctx.service_hash, transaction.elapsed, ctx.error != 0);
            }

            if(conf.trace_referer_enabled && ctx.referer!=0) {
                StatTranxReferer.add(ctx.referer, ctx.service_hash, transaction.elapsed, ctx.error != 0);
            }

            if(conf.stat_mtrace_enabled && ctx.mcaller_pcode!=0) {
                var key = StatTranxMtCaller.getService(ctx.mcaller_pcode, ctx.mcaller_okind, ctx.mcaller_spec, ctx.mcaller_url,ctx.service_hash);
                transaction.mthis_spec = conf.mtrace_spec_hash

                if (ctx.mcaller_spec && ctx.mcaller_spec!='') {
                    transaction.mcaller_spec = key.mcaller_spec = HashUtil.hashFromString(ctx.mcaller_spec);
                    DataTextAgent.MTRACE_SPEC.add(transaction.mcaller_spec, ctx.mcaller_spec);
                }
                if (ctx.mcaller_url  && ctx.mcaller_url!='') {
                    transaction.mcaller_url = key.mcaller_url = HashUtil.hashFromString(ctx.mcaller_url);
                    DataTextAgent.MTRACE_CALLER_URL.add(transaction.mcaller_url, ctx.mcaller_url);
                }

                // additional multi transaction
                //key.url = ctx.service_hash;
                var tc = StatTranxMtCaller.getService(key);
                if (tc != null) {
                    tc.count++;
                    if (transaction.errorLevel >= EventLevel.WARNING) {
                        tc.error++;
                    }
                    tc.time += transaction.elapsed;
                }

                // StatTranxMtCaller.add(ctx.mcaller_pcode,
                //      ctx.mcaller_okind,
                //      transaction.mcaller_spec,
                //      transaction.mcaller_url,
                //      ctx.service_hash,
                //      transaction.elapsed,
                //      ctx.error != 0);
            }

            var stat = StatTranX.getService(transaction.service);

            if(stat !== null) {
                stat.count++;
                if(transaction.error.isZero() ==false) {
                    stat.error++;
                }
                if(transaction.elapsed < 0) {
                    transaction.elapsed = 0;
                }

                stat.actived += ctx.profileActive;
                stat.time_sum += transaction.elapsed;
                if (transaction.elapsed > stat.time_max) {
                    stat.time_max = transaction.elapsed;
                }

                switch (transaction.apdex) {
                    case 2:
                        stat.apdex_satisfied++;
                        break;
                    case 1:
                        stat.apdex_tolerated++;
                        break;
                }
                if (stat.time_min==0 || transaction.elapsed < stat.time_min) {
                    stat.time_min = transaction.elapsed;
                }
                stat.time_std += (Long.fromInt(transaction.elapsed) * Long.fromInt(transaction.elapsed));

                stat.sql_count += transaction.sqlCount;
                stat.sql_time +=transaction.sqlTime;
                stat.sql_fetch += ctx.rs_count;
                stat.sql_fetch_time +=ctx.rs_time;

                stat.httpc_count += transaction.httpcCount;
                stat.httpc_time +=transaction.httpcTime;

                stat.malloc_sum +=transaction.malloc;
                stat.cpu_sum +=transaction.cpuTime;

                if (rejected) {
                    var now = Date.now();
                    if (now < this.last_reject + 1000) {
                        return;
                    }
                    this.last_reject = now;
                } else if (stat.profiled == true    // 이전(5분구간 내)에 프로파일이 수집된점이 있음
                    && ctx.profileActive == 0       // 액티브 스택을 추적한적이 없음
                    && transaction.elapsed < conf.profile_basetime
                    && transaction.error.isZero()) {
                    return;
                }
                stat.profiled = true;
            }

            var steps = ctx.profile.getSteps();
            profile.setProfile(steps);
            if(profile_zip_enabled){
                ZipProfile.getInstance().add(profile);
            } else {
                DataPackSender.sendProfilePack(profile);
            }

        } catch(e) {
            Logger.printError('WHATAP-199', 'dataprofile-agent error', e);
        }
    }
};

module.exports = DataProfileAgent;