/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var CounterTask = require('./counter-task'),
    MeterService = require('../meter/meter-service').MeterService,
    MeterActiveX = require('../meter/meter-activex'),
    DataPackSender = require('../../data/datapack-sender'),
    IntKeyLinkedMap = require('../../util/intkey-linkedmap'),
    LinkedMap = require('../../util/linkedmap'),
    conf = require('../../conf/configure'),
    Long = require('long');

function Item(tps, rtime) {
    this.tps = tps || 0;
    this.rtime = rtime || 0;
}

function Service() {
    CounterTask.call(this);
    this.avg30 = new IntKeyLinkedMap(7, 1).setMax(6);

    this.calcTpsResp = function (p, basetime /*Long*/ ) {
        // var basetime = basetime.toString(); /*Long to String*/
        var t = 0,
            r = 0,
            r90 = 0,
            c = 0,
            en = this.avg30.entries(),
            key, val;

        while (en.hasMoreElements()) {
            var entry = en.nextElement();
            key = entry.getKey();
            val = entry.getValue();
            if (key >= basetime) {
                c++;
                t += val.tps;
                r += val.rtime;
            }
        }

        if (c > 0) {
            p.tps = parseFloat(t / c);
            p.resp_time = parseInt(r / c);
        }
    };
};
Service.prototype = new CounterTask();
Service.prototype.constructor = Service;
Service.prototype.process = function (p) {

    var b = MeterService.getBucket();
    p.service_count = b.count;
    p.service_time = b.timesum;
    p.service_error = b.error;

    p.apdex_satisfied = b.count_satisfied;
    p.apdex_tolerated = b.count_tolerated;
    p.apdex_total = b.count;

    if (conf.tx_caller_meter_enabled) {
        tx_caller_meter(p);
    }
    if (conf.actx_meter_enabled) {
        tx_caller_active_meter(p);
    }

    if (p.duration > 0) {
        if (b.count <= 0) {
            this.avg30.put(p.time, new Item(0, 0));
        } else {
            this.avg30.put(p.time, new Item(b.count / p.duration, b.timesum / b.count));
        }
        this.calcTpsResp(p, p.time - 30000);
    }
    var hp = b.hitmap;

    b.reset();
    hp.time = p.time;

    DataPackSender.sendHitMapPack1(hp);
};

function createLinkedMap() {
    var m = new LinkedMap(309, 1).setMax(307);
    m.hash = function (k) {
        if (k._hash_) {
            return k._hash_;
        }
        var result = 1;
        result = 31 * result + k.pcode;
        result = 31 * result + k.okind;

        k._hash_ = result;
        return result;
    };
    m.equals = function (k1, k2) {
        return k1.pcode === k2.pcode && k1.okind === k2.okind;
    };
    return m;
}
function tx_caller_meter(p) {
    var meter = MeterService;
    if (meter.stat.size() > 0) {
        p.txcaller_oid_meter = new IntKeyLinkedMap();
        var en = meter.stat.keys();
        for (var i = 0; i < 300 && en.hasMoreElements(); i++) {
            var key = en.nextElement();
            var b = meter.stat.get(key);
            p.txcaller_oid_meter.put(key, {
                time: b.timesum,
                count: b.count,
                error: b.error,
                actx: 0
            });
        }
    }
    if (meter.statByGroup.size() > 0) {
        p.txcaller_group_meter = createLinkedMap();
        var en = meter.statByGroup.keys();
        for (var i = 0; i < 300 && en.hasMoreElements(); i++) {
            var key = en.nextElement();
            var b = meter.statByGroup.get(key);
            p.txcaller_group_meter.put(key, {
                time: b.timesum,
                count: b.count,
                error: b.error,
                actx: 0
            });
        }
    }
    p.txcaller_unknown = {
        count: meter.unknown.count,
        error: meter.unknown.error,
        time: meter.unknown.timesum,
        actx: 0
    };

    meter.resetStat();

};

function tx_caller_active_meter(p) {
    var meter = MeterActiveX;
    if (meter.statByOid.size() > 0) {
        if (p.txcaller_oid_meter == null) {
            p.txcaller_oid_meter = new IntKeyLinkedMap();
        }
        var en = meter.statByOid.keys();
        for (var i = 0; i < 300 && en.hasMoreElements(); i++) {
            var key = en.nextElement();
            var b = meter.statByOid.get(key);
            var org = p.txcaller_oid_meter.get(key);
            if (org) {
                org.actx = b;
            } else {
                p.txcaller_oid_meter.put(key, {
                    time: 0,
                    count: 0,
                    error: 0,
                    actx: b
                });
            }
        }
    }
    if (meter.statByGroup.size() > 0) {
        if (p.txcaller_group_meter == null) {
            p.txcaller_group_meter = createLinkedMap();
        }
        var en = meter.statByGroup.keys();
        for (var i = 0; i < 300 && en.hasMoreElements(); i++) {
            var key = en.nextElement();
            var b = meter.statByGroup.get(key);
            var org = p.txcaller_group_meter.get(key);
            if (org) {
                org.actx = b;
            } else {
                p.txcaller_group_meter.put(key, {
                    time: 0,
                    count: 0,
                    error: 0,
                    actx: b
                });
            }
        }
    }
    if(p.txcaller_unknown){
        p.txcaller_unknown.actx = meter.unknown;
    } else {
        p.txcaller_unknown = {
            count: 0,
            error: 0,
            time: 0,
            actx :meter.unknown
        };
    };
};

module.exports = Service;