var http = require('http');
/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var CounterTask = require('../../counter-task'),
    conf = require('../../../../conf/configure'),
    Logger = require('../../../../logger'),
    DateUtil = require('../../../../util/dateutil');

function AwsEcsClientThread() {
    if (typeof AwsEcsClientThread.instance === "object") {
        return AwsEcsClientThread.instance;
    }

    this.millicore = 0;
    this.last_milli_check = 0;
    this.check = AwsEcsClientThread.CHECK_MAX;


    this.cpu = 0;
    this.cpu_sys = 0;
    this.cpu_user = 0;
    this.mem_pct = 0;

    var self = this;
    setTimeout(function () {
        self.run();
    }, 100);
    AwsEcsClientThread.instance = this;
};
AwsEcsClientThread.prototype = new AwsEcsClientThread();

AwsEcsClientThread.prototype.run = function () {
    var self = this;
    try {
        if (conf.aws_ecs_enabled) {
            self.intervalIndex = setInterval(function () {
                var meta_uri = process.env["ECS_CONTAINER_METADATA_URI"];
                if (meta_uri != null) {
                    self.getCpuLimit(meta_uri);
                    self.process(meta_uri);
                }
            }, conf.getProperty('aws_ecs_stat_interval', 3000));
        }
    } catch (e) {
        Logger.printError('WHATAP ECS-ERR ', 'run exception', e, true);
    }
}

AwsEcsClientThread.prototype.stop = function () {
    if (this.intervalIndex) {
        clearInterval(this.intervalIndex);
        this.intervalIndex = undefined;
    }
}

AwsEcsClientThread.prototype.getCpuLimit = function (meta_uri) {
    var self = this;

    if (this.check < 0)
        return;
    var now = Date.now();
    if (now - DateUtil.MILLIS_PER_HOUR * 8 < this.last_milli_check)
        return;

    // var json = JSON.parse('{"DockerId":"e9950915-abaa-467f-9898-3bd0d4e6d0aa","Name":"web-application","DockerName":"ecs-web-application-1","Image":"nginx:latest","ImageID":"sha256:123456789abcdef...","Labels":{"com.amazonaws.ecs.cluster":"default","com.amazonaws.ecs.task-arn":"arn:aws:ecs:us-east-1:123456789012:task/default/1234567890abcdef","com.amazonaws.ecs.task-definition-family":"web-application","com.amazonaws.ecs.task-definition-version":"1"},"DesiredStatus":"RUNNING","KnownStatus":"RUNNING","Limits":{"CPU":256,"Memory":512},"CreatedAt":"2024-01-01T00:00:00.000Z","StartedAt":"2024-01-01T00:00:01.000Z","Type":"NORMAL","Networks":[{"NetworkMode":"awsvpc","IPv4Addresses":["172.31.10.100"],"AttachmentIndex":0,"MACAddress":"0a:1b:2c:3d:4e:5f","IPv4SubnetCIDRBlock":"172.31.0.0/20","DomainNameServers":["169.254.169.253"],"DomainNameSearchList":["us-east-1.compute.internal"],"PrivateDNSName":"ip-172-31-10-100.us-east-1.compute.internal","SubnetGatewayIpv4Address":"172.31.0.1/20"}],"ContainerARN":"arn:aws:ecs:us-east-1:123456789012:container/1234567890abcdef","LogDriver":"awslogs","LogOptions":{"awslogs-group":"/ecs/web-application","awslogs-region":"us-east-1","awslogs-stream":"ecs/web-application/1234567890abcdef"}}');
    // 2025-01-23 ECS fargate
    // var json = JSON.parse('{"DockerId":"a663f01570b14228b0795db9c8d11e16-2785547837","Name":"test_container","DockerName":"test_container","Image":"whataphub/ecs_sample:v0.0.4","ImageID":"sha256:748b5cd202288478bbdbd4baf8fe17c8b660163c24c513db5ccbe7a1bdcae0e0","Labels":{"com.amazonaws.ecs.cluster":"arn:aws:ecs:ap-northeast-2:592247757306:cluster/ecs_sample","com.amazonaws.ecs.container-name":"test_container","com.amazonaws.ecs.task-arn":"arn:aws:ecs:ap-northeast-2:592247757306:task/ecs_sample/a663f01570b14228b0795db9c8d11e16","com.amazonaws.ecs.task-definition-family":"ecs_task_sample","com.amazonaws.ecs.task-definition-version":"4"},"DesiredStatus":"RUNNING","KnownStatus":"RUNNING","Limits":{"CPU":2},"CreatedAt":"2025-01-23T04:42:03.843875803Z","StartedAt":"2025-01-23T04:42:03.843875803Z","Type":"NORMAL","Networks":[{"NetworkMode":"awsvpc","IPv4Addresses":["172.31.63.147"]}]}');
    // if (json && json["Limits"]) {
    //     this.millicore = json.Limits.CPU;
    //     if (this.millicore < 128) {
    //         this.millicore *= 1024;
    //     }
    //     Logger.print("SysCPU", " ECS CPU Limits : " + (this.millicore / 1024) + "core", false)
    // }

    read(meta_uri, function(json){
        if (json && json["Limits"]) {
            this.millicore = json.Limits.CPU;
            if (this.millicore < 128) {
                this.millicore *= 1024;
            }
            Logger.print("WHATAP-811", "ECS CPU Limits : " + (this.millicore / 1024) + "core", false)
        }
    }, function(){
        self.millicore = 0;
        if (--self.check >= 0) {
            Logger.printError('WHATAP-819' , 'not receive limit cpu', e , true);
        }
    })
}

AwsEcsClientThread.prototype.process = function (meta_uri) {
    var self = this;

    fetch(meta_uri + '/stats', {timeout: 3000})
        .then(response => response.text())
        .then(stats => {
            if (!stats || stats.length === 0) {
                self.reset();
                return;
            }

            // let stats = '{"blkio_stats":{"io_merged_recursive":[],"io_queue_recursive":[],"io_service_bytes_recursive":[{"major":259,"minor":3,"op":"Read","value":0},{"major":259,"minor":3,"op":"Write","value":45514752},{"major":259,"minor":3,"op":"Sync","value":4268032},{"major":259,"minor":3,"op":"Async","value":41246720},{"major":259,"minor":3,"op":"Discard","value":0},{"major":259,"minor":3,"op":"Total","value":45514752}],"io_service_time_recursive":[],"io_serviced_recursive":[{"major":259,"minor":3,"op":"Read","value":0},{"major":259,"minor":3,"op":"Write","value":342},{"major":259,"minor":3,"op":"Sync","value":272},{"major":259,"minor":3,"op":"Async","value":70},{"major":259,"minor":3,"op":"Discard","value":0},{"major":259,"minor":3,"op":"Total","value":342}],"io_time_recursive":[],"io_wait_time_recursive":[],"sectors_recursive":[]},"cpu_stats":{"cpu_usage":{"percpu_usage":[5585610339,5944268076],"total_usage":15373171220,"usage_in_kernelmode":2150000000,"usage_in_usermode":7990000000},"online_cpus":2,"throttling_data":{"periods":0,"throttled_periods":0,"throttled_time":0}},"id":"10b741d47dc649d5b321f83c7c14c950-441560335","memory_stats":{"limit":9223372036854772000,"max_usage":246165504,"stats":{"active_anon":0,"active_file":27439104,"cache":56365056,"dirty":0,"hierarchical_memory_limit":2147483648,"hierarchical_memsw_limit":9223372036854772000,"inactive_anon":98267136,"inactive_file":28925952,"mapped_file":0,"pgfault":303666,"pgmajfault":0,"pgpgin":199749,"pgpgout":161965,"rss":98402304,"rss_huge":0,"total_active_anon":0,"total_active_file":27439104,"total_cache":56365056,"total_dirty":0,"total_inactive_anon":98267136,"total_inactive_file":28925952,"total_mapped_file":0,"total_pgfault":303666,"total_pgmajfault":0,"total_pgpgin":199749,"total_pgpgout":161965,"total_rss":98402304,"total_rss_huge":0,"total_unevictable":0,"total_writeback":135168,"unevictable":0,"writeback":135168},"usage":168361984},"name":"simple-user-app","networks":{"eth1":{"rx_bytes":1604876131,"rx_dropped":0,"rx_errors":0,"rx_packets":1076062,"tx_bytes":6252490,"tx_dropped":0,"tx_errors":0,"tx_packets":74457}},"num_procs":0,"pids_stats":{},"precpu_stats":{"cpu_usage":{"percpu_usage":[5570558410,5925648831],"total_usage":15328276321,"usage_in_kernelmode":2140000000,"usage_in_usermode":7970000000},"online_cpus":2,"throttling_data":{"periods":0,"throttled_periods":0,"throttled_time":0}},"preread":"2023-05-30T11:29:20.720196524Z","read":"2023-05-30T11:29:30.719936356Z","storage_stats":{}}';
            // 2025-01-23 ECS fargate
            // var stats = '{"read":"2025-01-23T04:42:03.844081622Z","preread":"0001-01-01T00:00:00Z","pids_stats":{},"blkio_stats":{"io_service_bytes_recursive":[],"io_serviced_recursive":[],"io_queue_recursive":[],"io_service_time_recursive":[],"io_wait_time_recursive":[],"io_merged_recursive":[],"io_time_recursive":[],"sectors_recursive":[]},"num_procs":0,"storage_stats":{},"cpu_stats":{"cpu_usage":{"total_usage":33451672,"percpu_usage":[22728826,10722846],"usage_in_kernelmode":10000000,"usage_in_usermode":20000000},"system_cpu_usage":397640000000,"online_cpus":2,"throttling_data":{"periods":0,"throttled_periods":0,"throttled_time":0}},"precpu_stats":{"cpu_usage":{"total_usage":0,"usage_in_kernelmode":0,"usage_in_usermode":0},"throttling_data":{"periods":0,"throttled_periods":0,"throttled_time":0}},"memory_stats":{"usage":765952,"max_usage":5267456,"stats":{"active_anon":0,"active_file":0,"cache":0,"dirty":0,"hierarchical_memory_limit":4294967296,"hierarchical_memsw_limit":9223372036854771712,"inactive_anon":135168,"inactive_file":0,"mapped_file":0,"pgfault":2112,"pgmajfault":0,"pgpgin":1452,"pgpgout":1452,"rss":270336,"rss_huge":0,"total_active_anon":0,"total_active_file":0,"total_cache":0,"total_dirty":0,"total_inactive_anon":135168,"total_inactive_file":0,"total_mapped_file":0,"total_pgfault":2112,"total_pgmajfault":0,"total_pgpgin":1452,"total_pgpgout":1452,"total_rss":270336,"total_rss_huge":0,"total_unevictable":0,"total_writeback":0,"unevictable":0,"writeback":0},"limit":9223372036854771712},"name":"test_container","id":"a663f01570b14228b0795db9c8d11e16-2785547837","networks":{"eth1":{"rx_bytes":463375353,"rx_packets":310281,"rx_errors":0,"rx_dropped":0,"tx_bytes":250514,"tx_packets":2681,"tx_errors":0,"tx_dropped":0}}}';
            // var stats = '{"read":"2025-01-23T04:52:53.853606208Z","preread":"2025-01-23T04:52:43.853359478Z","pids_stats":{},"blkio_stats":{"io_service_bytes_recursive":[],"io_serviced_recursive":[],"io_queue_recursive":[],"io_service_time_recursive":[],"io_wait_time_recursive":[],"io_merged_recursive":[],"io_time_recursive":[],"sectors_recursive":[]},"num_procs":0,"storage_stats":{},"cpu_stats":{"cpu_usage":{"total_usage":8111895890,"percpu_usage":[3973083470,4138812420],"usage_in_kernelmode":2220000000,"usage_in_usermode":5910000000},"system_cpu_usage":1692450000000,"online_cpus":2,"throttling_data":{"periods":0,"throttled_periods":0,"throttled_time":0}},"precpu_stats":{"cpu_usage":{"total_usage":8023476115,"percpu_usage":[3922781879,4100694236],"usage_in_kernelmode":2180000000,"usage_in_usermode":5820000000},"system_cpu_usage":1672470000000,"online_cpus":2,"throttling_data":{"periods":0,"throttled_periods":0,"throttled_time":0}},"memory_stats":{"usage":66977792,"max_usage":98480128,"stats":{"active_anon":0,"active_file":0,"cache":0,"dirty":0,"hierarchical_memory_limit":4294967296,"hierarchical_memsw_limit":9223372036854771712,"inactive_anon":61906944,"inactive_file":0,"mapped_file":0,"pgfault":463386,"pgmajfault":0,"pgpgin":147477,"pgpgout":132323,"rss":62042112,"rss_huge":0,"total_active_anon":0,"total_active_file":0,"total_cache":0,"total_dirty":0,"total_inactive_anon":61906944,"total_inactive_file":0,"total_mapped_file":0,"total_pgfault":463386,"total_pgmajfault":0,"total_pgpgin":147477,"total_pgpgout":132323,"total_rss":62042112,"total_rss_huge":0,"total_unevictable":0,"total_writeback":0,"unevictable":0,"writeback":0},"limit":9223372036854771712},"name":"test_container","id":"a663f01570b14228b0795db9c8d11e16-2785547837","networks":{"eth1":{"rx_bytes":463489958,"rx_packets":311334,"rx_errors":0,"rx_dropped":0,"tx_bytes":1962223,"tx_packets":4432,"tx_errors":0,"tx_dropped":0}}}'
            const o = JSON.parse(stats);

            if (conf.getProperty('debug_aws_ecs_enabled', false)) {
                Logger.print("AwsEcsClient", 30, "meta_uri=" + meta_uri + " stats\n" + JSON.stringify(o, null, 4));
            }

            const precpu_stats = o.precpu_stats || {};
            const cpu_stats = o.cpu_stats || {};
            const memory_stats = o.memory_stats || {};

            const system_cpu_usage = delta(precpu_stats, cpu_stats, "system_cpu_usage");

            if (system_cpu_usage > 0) {
                const precpu_usage = precpu_stats.cpu_usage || {};
                const cpu_usage = cpu_stats.cpu_usage || {};

                const usage_tot = delta(precpu_usage, cpu_usage, "total_usage");
                const usage_sys = delta(precpu_usage, cpu_usage, "usage_in_kernelmode");
                const usage_usr = delta(precpu_usage, cpu_usage, "usage_in_usermode");

                self.cpu = (usage_tot * 100.0) / system_cpu_usage;
                self.cpu_sys = (usage_sys * 100.0) / system_cpu_usage;
                self.cpu_user = (usage_usr * 100.0) / system_cpu_usage;
            } else {
                self.reset();
            }

            if (memory_stats) {
                const mem_usage = memory_stats.usage || 0;
                const memory_stats_stats = memory_stats.stats;

                if (memory_stats_stats) {
                    const mem_limit = memory_stats_stats.hierarchical_memory_limit || 0;
                    self.mem_pct = (mem_usage * 100.0) / mem_limit;
                } else {
                    self.mem_pct = 0;
                }
            } else {
                self.mem_pct = 0;
            }
        })
        .catch(error => {
            Logger.printError("WHATAP-819", "ECS Process error: " + error, false)
            self.reset();
        });
};

AwsEcsClientThread.prototype.reset = function () {
    this.cpu = 0;
    this.cpu_sys = 0;
    this.cpu_user = 0;
};

// reset 함수 추가
AwsEcsClientThread.prototype.reset = function () {
    this.cpu = 0;
    this.cpu_sys = 0;
    this.cpu_user = 0;
};

AwsEcsClientThread.CHECK_MAX = 3;

function read(url, succ, fail) {
    http.get(url, function (res) {
        var statusCode = res.statusCode;

        if (statusCode !== 200) {
            if (fail) fail();

            Logger.printError('WHATAP-999', 'read http call (not 200) ', e, true);
            return;
        }

        res.setEncoding('utf8');

        var rawData = '';
        res.on('data', function (chunk) {
            rawData += chunk;
        });
        res.on('end', function () {
            try {
                var stats = JSON.parse(rawData);
                if (succ) succ(stats);
            } catch (e) {
                if (fail) fail(e);
                Logger.printError('WHATAP-999', 'read http call (not application/json) ', e, true);
            }
        });
    }).on('error', (e) => {
        if (fail) fail(e);
        Logger.printError('WHATAP-999', 'read http call ', e, true);
    });
}

function delta(j1, j2, key) {
    if (!j1 || !j2)
        return 0;

    var v1 = j1[key];
    var v2 = j2[key];

    if(v1 && v2)
        return v2 - v1;
    else
        return 0;
}

module.exports = new AwsEcsClientThread();
