/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

function Meter() {
	this.BUCKET_SIZE = 301;
	this.initTime = Math.round(Date.now() / 1000);
	this.lastTime = this.initTime;
	this.lastOffset = this.initTime % this.BUCKET_SIZE;
	// this.clear = function() {};
};
Meter.prototype.getPosition = function() {
	var currentTime = Math.round(Date.now() / 1000);
	if (currentTime != this.lastTime) {
		for (i = 0; i < (currentTime - this.lastTime) && i < this.BUCKET_SIZE; i++) {
			this.lastOffset = (this.lastOffset + 1 >= this.BUCKET_SIZE) ? 0
					: this.lastOffset + 1;
			this.clear(this.lastOffset);
		}
		this.lastTime = currentTime;
		this.lastOffset = this.lastTime % this.BUCKET_SIZE;
	}

	return this.lastOffset;

};

Meter.prototype.check = function(period) {
	if (period >= this.BUCKET_SIZE)
		period = this.BUCKET_SIZE - 1;
	return period;
};

Meter.prototype.getTime = function() {
	return new Date().getTme() / 1000;
};

Meter.prototype.desc = function(pos) {
	if (pos === 0)
		pos = this.BUCKET_SIZE - 1;
	else
		pos--;
	return pos;
};

function MeterService() {
	if (typeof MeterService.instance === "object") {
		return MeterService.instance;
	}
	this.txCount = new Array(this.BUCKET_SIZE);
	this.errCount = new Array(this.BUCKET_SIZE);
	this.txDuration = new Array(this.BUCKET_SIZE);

	for ( var i = 0; i < this.BUCKET_SIZE; i++) {
		this.txCount[i] = 0;
		this.errCount[i] = 0;
		this.txDuration[i] = 0;
	}
	MeterService.instance = this;
};

MeterService.prototype = new Meter();

MeterService.prototype.clear = function(offset) {
	this.txCount[offset] = 0;
	this.errCount[offset] = 0;
	this.txDuration[offset] = 0;
};

MeterService.prototype.add = function(elapsed, err) {

	var pos = this.getPosition();
	this.txCount[pos]++;
	this.txDuration[pos] += elapsed;
	if (err)
		this.errCount[pos]++;

};

MeterService.prototype.getTPS = function(period) {
	period = this.check(period);
	var pos = this.getPosition();

	var sum = 0;
	for ( var i = 0; i < period; i++, pos = this.desc(pos)) {
		sum += this.txCount[pos];
	}
	return (sum / period);
};

MeterService.prototype.getElapsedTime = function(period) {
	period = this.check(period);
	var pos = this.getPosition();

	var sum = 0;
	var cnt = 0;
	for ( var i = 0; i < period; i++, pos = this.desc(pos)) {
		sum += this.txDuration[pos];
		cnt += this.txCount[pos];
	}
	return ((cnt === 0) ? 0 : sum / cnt);
};

MeterService.prototype.getError = function(period) {
	period = this.check(period);
	var pos = this.getPosition();

	var cnt = 0;
	var err = 0;
	for ( var i = 0; i < period; i++, pos = this.desc(pos)) {
		cnt += this.txCount[pos];
		err += this.errCount[pos];
	}
	return ((cnt === 0) ? 0 : ((err / cnt) * 100.0));
};

MeterService.prototype.getServiceCount = function(period) {
	period = this.check(period);
	var pos = this.getPosition();

	var sum = 0;
	for ( var i = 0; i < period; i++, pos = this.desc(pos)) {
		sum += this.txCount[pos];
	}
	return sum;
};

MeterService.prototype.getServiceError = function(period) {
	period = this.check(period);
	var pos = this.getPosition();

	var sum = 0;
	for ( var i = 0; i < period; i++, pos = this.desc(pos)) {
		sum += this.errCount[pos];
	}
	return sum;
};

function MeterResource() {
	this.value = new Array(this.BUCKET_SIZE);
	this.count = new Array(this.BUCKET_SIZE);

	for ( var i = 0; i < this.BUCKET_SIZE; i++) {
		this.value[i] = 0;
		this.count[i] = 0;

	}

};
MeterResource.prototype = new Meter();

MeterResource.prototype.clear = function(offset) {
	this.value[offset] = 0;
	this.count[offset] = 0;
};

MeterResource.prototype.add = function(value) {
	var pos = this.getPosition();
	this.value[pos] += value;
	this.count[pos]++;

};

MeterResource.prototype.getAvg = function(period) {
	var period = this.check(period);
	var pos = this.getPosition();
	var count = 0;
	var sum = 0;

	for ( var i = 0; i < period; i++, pos = this.desc(pos)) {
		sum += this.value[pos];
		count += this.count[pos];
	}
	if (count === 0) {
		return 0;
	}
	return sum / count;
};

module.exports.MeterService = MeterService;
module.exports.MeterResource = MeterResource;
