/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var CounterPack = require('../pack/counter-pack'),
    Service = require('./task/service'),
    AgentInfo = require('./task/agentinfo'),
    ActiveTransaction = require('./task/activetransaction'),
    ProcCpu = require('./task/proc-cpu'),
    ResSysCpu = require('./task/res-sys-cpu'),
    HeapMem = require('./task/heapmem'),
    SystemPerf = require('./task/systemperf'),
    RealtimeUser = require('./task/realtimeuser'),
    Socketio = require('./task/socketio'),
    GCStat = require('./task/gcstat'),
    Sql = require('./task/sql'),
    HttpC = require('./task/httpc'),
    StatError = require('../stat/stat-error'),
    TagCounterPack = require('../pack/tagcount-pack'),
    TraceContextManager = require('../trace/trace-context-manager'),
    secuMaster = require('../net/security-master'),
    DataPackSender = require('../data/datapack-sender'),
    conf = require('../conf/configure'),
    DateUtil = require('./../util/dateutil'),
    KubeClient       = require('../kube/kube-client'),
    KubeUtil            = require('../util/kube-util'),
    Logger              = require('../logger');

function CounterManager(agent) {
    this.agent = agent;
    this.time = Date.now();
    
    this.intervalIndex = undefined;
};
CounterManager.plugin={}
CounterManager.prototype.run = function () {
    var self = this;

    process.nextTick(function () {
        var tasks = [];
        tasks.push(new Service());
        tasks.push(new AgentInfo());
        tasks.push(new ActiveTransaction());
        tasks.push(new RealtimeUser());
        tasks.push(new HeapMem());
        tasks.push(new ProcCpu());
        tasks.push(new ResSysCpu());
        tasks.push(new SystemPerf());
        tasks.push(new Socketio());
        tasks.push(new GCStat());
        tasks.push(new Sql());
        tasks.push(new HttpC());
        self.intervalIndex = setInterval(function(){
            self.process(tasks);    
        },5000);
        self.process(tasks);
    });

    if(conf.whatap_micro_enabled || process.env.WHATAP_MICRO_ENABLED === "true"){
        KubeUtil.loadContainerId();
        if (conf.enabled_master_agent_call) KubeClient.run();
    }
};
CounterManager.prototype.stop = function(){
    if(this.intervalIndex){
        clearInterval(this.intervalIndex);
        this.intervalIndex = undefined;
    }
}

CounterManager.prototype.process = function (tasks) {
    var interval = conf.counter_interval || 5000,
        len = tasks.length, self = this, task;

    var now = parseInt(Date.now() / 1000),
        interval_sec = interval / 1000;

    if (now != parseInt(now / interval_sec) * interval_sec) {
        setTimeout(function () {
            self.process(tasks);
        }, 1000, 'whatap');
        return;
    }

    var enumer = TraceContextManager.getContextEnumeration();
    while (enumer.hasMoreElements()) {
        var ctx = enumer.nextElement();
        if (ctx.getElapsedTime() > conf.profile_max_time) {
            //TraceContextManager.end(ctx._id);
            if(conf.getProperty("ignore_http_lost_connection", false) === true){
                Logger.print("WHATAP-072", "Lost Connection was ignored, " + ctx._id, false);
                TraceContextManager.end(ctx._id);
                ctx = null;
            }else{
                // var obj = TraceContextManager.resume(ctx._id);
                TraceContextManager.addStep("Lost Connection","", ctx);
                //ctx.status = 5;
                //ctx.error = StatError.addError(500, "Lost Connection", ctx.service_hash);
                TraceContextManager.endTrace(ctx);
            }
        }
    }

    if (conf.counter_enabled) {
        var pack = new CounterPack();
        pack.pcode = secuMaster.PCODE;
        pack.oid = secuMaster.OID;
        pack.time = Math.floor(DateUtil.currentTime() / 5000) * 5000;
        pack.duration = Math.floor(interval / 1000);
        for (var i = 0; i < len; i++) {
            task = tasks[i];
            try{
                task.process(pack);
            } catch (e) {
                Logger.printError('WHATAP-070', 'Counter Manager process ', e, true);
            }
        }
        pack.starttime = this.agent.starttime;

        try {
            if (CounterManager.plugin["extra"]) {
                CounterManager.plugin["extra"](pack);
            }
        } catch (e) {
            Logger.printError('WHATAP-071', 'Counter Manager Plugin ', e, true);
        }
        DataPackSender.sendCounterPack(pack);
    }
};

module.exports = CounterManager;