/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var os = require('os'),
    fs = require('fs'),
    path = require('path');

const RequestLog = require('../requestlog');

var Interceptor      = require('./interceptor').Interceptor,
    HttpObserver     = require('../observers/http-observer').HttpObserver,
    NetObserver      = require('../observers/net-observer').NetObserver,
    ClusterObserver  = require('../observers/cluster-observer').ClusterObserver,
    ExpressObserver  = require('../observers/express-observer').ExpressObserver,
    GlobalObserver   = require('../observers/global-observer').GlobalObserver,
    MysqlObserver    = require('../observers/mysql-observer').MysqlObserver,
    MariaObserver    = require('../observers/maria-observer').MariaObserver,
    SocketioObserver = require('../observers/socket.io-observer').SocketIOObserver,
    WebsocketObserver = require('../observers/websocket-observer').WebsocketObserver,
    ProcessObserver  = require('../observers/process-observer').ProcessObserver,
    FileObserver     = require('../observers/file-observer').FileObserver,
    MongoObserver    = require('../observers/mongo-observer').MongoObserver,
    MongooseObserver = require('../observers/mongoose-observer').MongooseObserver,
    RedisObserver    = require('../observers/redis-observer').RedisObserver,
    MssqlObserver    = require('../observers/mssql-observer').MssqlObserver,
    ThriftObserver    = require('../observers/thrift-observer').ThriftObserver,
    PromiseObserver  = require('../observers/promise-observer').PromiseObserver,
    PgSqlObserver    = require('../observers/pgsql-observer').PgSqlObserver,
    ScheduleObserver = require('../observers/schedule-observer').ScheduleObserver,
    GRpcObserver     = require('../observers/grpc-observer').GRpcObserver,
    ApolloObserver   = require('../observers/apollo-server-observer').ApolloServerObserver;


var Configuration    = require('./../conf/configure'),
    SecurityMaster   = require('./../net/security-master'),
    TcpSession       = require('./../net/tcp-session'),
    PackageCtrHelper = require('./../control/packagectr-helper'),
    DataPackSender   = require('../data/datapack-sender'),
    CounterManager   = require('./../counter/counter-manager'),
    NodeUtil         = require('./../util/nodeutil'),
    DateUtil         = require('./../util/dateutil'),
    WhatapUtil       = require('./../util'),
    ParamPack        = require('../pack/param-pack').ParamPack,
    PluginLoaderManager = require('../plugin/plugin-loadermanager'),
    TraceContextManager = require('../trace/trace-context-manager'),
    DataTextAgent    = require('../data/datatext-agent').agent,
    TextTypes       = require('../lang/text-types'),
    HashUtil            = require('../util/hashutil'),
    MessageStep         = require('../step/message-step'),
    Logger           = require('../logger');

require('../stat/stat-tranx');
require('../stat/stat-sql');
require('../stat/stat-httpc');
require('../stat/stat-remote-ip');
require('../stat/stat-remote-ipurl');
require('../stat/stat-useragent');
require('../stat/timingsender');

var NodeAgent = function(opt) {
    this._userOpt = opt;
    this._initialized = false;
    this.aop = new Interceptor(this);
    this._conf = Configuration;
    this._securityMaster = SecurityMaster;
    this._tcpsession = null;
    this._counterManager = null;
    this.setLoopTime = 5000;
    this.connectCount = 0;
    this.userProfileHash = 0;
    this.init();
};

NodeAgent.prototype.plugin=function(name,func){
    if('extra'===name){
        //test
        CounterManager.plugin[name]=func;
    }
}

NodeAgent.prototype.findRoot = function () {
    var self = this;
    var root = process.cwd();
    while(root.length > 1) {
        var dir_name = path.join(root, 'package.json');
        if(fs.existsSync(dir_name)) {
            self._conf['app.root'] = root;
            return;
        }
        root = path.join(root, '..');
    }

    root = path.dirname(require.main.filename);
    while(root.length > 1) {
        var dir_name = path.join(root, 'package.json');
        if(fs.existsSync(dir_name)) {
            self._conf['app.root'] = root;
            return;
        }
        root = path.join(root, '..');
    }
};
NodeAgent.prototype.init = function(cb) {
    var self = this;
    if(self._initialized) {
        return;
    }
    self._initialized = true;
    self.starttime = Date.now();

    self.findRoot();

    Logger.initializer.process();
    Logger.print('WHATAP-001', 'Start initialize WhaTap Agent... Root[' + self._conf['app.root'] + ']', true);

    if(self._conf['app.root'] == null || self._conf['app.root'].length == 0) {
        return Logger.print("WHATAP-001", "Can not find application root directory", true);
    }

    NodeUtil.getPackageJson();
    PackageCtrHelper.dynamicHook();

    self._conf.init(this._userOpt , function(e) {
        if(e) {
            Logger.printError("WHATAP-002", "Configuataion initialize error. " + e.message,
                e, true);
            return;
        }
        Logger.print('WHATAP-101', 'Finish initialize configuration... ' + Boolean(self._conf['reqlog_enabled']) , false);
        if(Boolean(self._conf['reqlog_enabled']) == true ) {
            Logger.print("WHATAP-REQLOG" , "ReqLog Init Start !!!! " , false);
            RequestLog.initializer.process();
        }

        self._securityMaster.run( function (err) {
            if(err) {
                Logger.printError('WHATAP-104', 'Failed to connect to whatap server', err, false);
                if(self._counterManager) self._counterManager.stop();
                return;
            }

            self.connect(function (err) {
                if(!err) {
                    self.initOK(cb);
                }
            });
        });
    });
    self.loadObserves();
    return this;
};

NodeAgent.prototype.initOK = function(cb) {
    var self = this;

    Logger.ONAME = self._securityMaster.ONAME;
    Logger.initializer.reset();

    TraceContextManager.initialized = true;
    DataTextAgent.reset();

    var param = new ParamPack();
    param.putString("whatap.version", NodeUtil.getVersion()+' '+ NodeUtil.getReleaseDate());
    param.putString("os.name", os.platform());
    param.putString("os.arch", os.arch());
    param.putString('os.release', os.release());
    param.putString('node.version', process.version);
    param.putString('node.uptime', (new Date()).toString());
    param.putString('node.name', NodeUtil.getName());
    param.putString('user.timezone', NodeUtil.getTimeZone());
    param.putString('user.home', os.homedir());
    param.putString('user.hostname', os.hostname());
    // CLOUD PLATFORM INFO
    param.putString('CLOUD_PLATFORM', WhatapUtil.cloudPlatformCheck());
    DataPackSender.sendBoot(param);

    if(self._counterManager === null) {
        self._counterManager = new CounterManager(self);
        self._counterManager.run();
        PluginLoaderManager.start();
    }

    WhatapUtil.printWhatap();
    self.connectCount = 0;
    self.setLoopTime = 5000;
    if(cb) cb();
}


NodeAgent.prototype.context = function(cb) {
    return TraceContextManager.getCurrentContext();
}

NodeAgent.prototype.connect = function (cb) {
    var self = this;
    if(self.connectCount >0 && self.connectCount % 10 == 0 ) {
        Logger.print("WHATAP-171",
            "[WhaTap Agent] Can not find server port.\n" +
            "Did you add requre('whatap') as the first line of the app's main module?\n" +
            "https://service.whatap.io/project/" + self._securityMaster.PCODE + "/install",
            true);
        if(self.setLoopTime == 5000) {
            self.setLoopTime += 25000;
            Logger.print("WHATAP-150", "[WhaTap Agent] setLoopTime Up -> " + self.setLoopTime , false)
        }else if(self.setLoopTime == 30000) {
            self.setLoopTime += 30000;
            Logger.print("WHATAP-150", "[WhaTap Agent] setLoopTime Up -> " + self.setLoopTime , false)
        } else if(self.setLoopTime == 60000) {
            self.setLoopTime += 300000;
            Logger.print("WHATAP-150", "[WhaTap Agent] setLoopTime Up -> " + self.setLoopTime , false)
        }
    }
    Logger.print("WHATAP-170", "[WhaTap Agent] now waiting for starting......", false)
    self._tcpsession = TcpSession;
    setTimeout(function () {
        Logger.print("WHATAP-011" , "Connecton Retry....")
        self.connectCount++;
        self._tcpsession.open(function(err) {
            if(err) {
                Logger.printError("WHTAP-11", "TCP SESSION OPEN ERROR", err ,false);
                return self.connect(cb);
            };
            if(cb) cb();
        });
    }, self.setLoopTime);
};

NodeAgent.prototype.connectCB = function (cb) {
    var self = this;
    if(self.connectCount >0 && self.connectCount % 5 == 0 ) {
        return cb(new Error());
    }

    Logger.print("WHATAP-970", "[WhaTap Agent] now waiting for starting......", true)
    self._tcpsession = TcpSession;
    setTimeout(function () {
        self._tcpsession.open(function(err) {
            if(err) {
                Logger.printError("WHTAP-911", "TCP SESSION OPEN ERROR", err ,true);
                self.connectCB(cb);
                return;
            };
            if(cb) cb();
        });
    }, self.setLoopTime);
};

NodeAgent.prototype.loadObserves = function() {
    var agent = this;
    var observes = [];

    observes.push(HttpObserver);
    observes.push(NetObserver);
    observes.push(ClusterObserver);
    observes.push(MysqlObserver);
    observes.push(MariaObserver);
    observes.push(SocketioObserver);
    observes.push(WebsocketObserver);
    observes.push(ExpressObserver);
    observes.push(FileObserver);
    observes.push(MongoObserver);
    observes.push(MongooseObserver);
    observes.push(RedisObserver);
    observes.push(MssqlObserver);
    observes.push(ThriftObserver);
    observes.push(PromiseObserver);
    observes.push(PgSqlObserver);
    observes.push(ScheduleObserver);
    // observes.push(GRpcObserver);
    observes.push(ApolloObserver);

    var packageToObserve = {};
    observes.forEach(function(observeObj) {
        var observe = new observeObj(agent);
        observe.packages.forEach(function(pkg){
            if(!packageToObserve[pkg]){
                packageToObserve[pkg] = [];
            }
            packageToObserve[pkg].push( observe );
        });
    });

    agent.aop.after(module.__proto__, 'require', function(obj, args, result) {
        var observes = packageToObserve[args[0]];

        if(observes) {
            observes.forEach(function(observe){
                return observe.inject(result, args[0]);
            });
        }
    });

    for (var name in packageToObserve) {
        try {
            require(name);
        } catch (err) {
            Logger.print("WHTAP-loadObserves", 'unable to load ' + name + ' module', false);
        }
    }
    new ProcessObserver(agent).inject(process, 'process');
    new GlobalObserver(agent).inject(global, 'global');
};

NodeAgent.prototype.setServicePort = function (port) {
    Configuration['whatap.port'] = port || 0;
}

NodeAgent.prototype.profile = function(desc, value){
    var self = this;

    var ctx = TraceContextManager.getCurrentContext();
    if(ctx == null) { return; }

    if( self.userProfileHash === 0){
        self.userProfileHash = HashUtil.hashFromString('CustomProfileStep');
    }
    DataTextAgent.add(TextTypes.MESSAGE,self.userProfileHash, 'CustomProfileStep');
    var step = new MessageStep();
    step.hash = self.userProfileHash;
    step.start_time = ctx.getElapsedTime();
    step.desc = desc;
    if(value){
        step.value=false;
    }
    ctx.profile.add(step);

}
// var plugins={};
// NodeAgent.prototype.plugin = function(name, func){
//     if(func){
//         plugins[name]=func;
//     }
//     return plugins[name];
// } 
exports.NodeAgent = new NodeAgent();