/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var fs      = require('fs'),
    path    = require('path');

var TraceContextManager = require('../trace/trace-context-manager'),
    DataTextAgent       = require('../data/datatext-agent'),
    MethodStep          = require('../step/method-stepx'),
    HashUtil            = require('../util/hashutil'),
    ArrayUtil           = require('../util/array-util'),
    conf                = require('../conf/configure'),
    Logger              = require('../logger');

var loadedPackageList   = [],
    loadedPackages      = {},
    dependencies        = [];

var PackageCtrHelper = function() {};
PackageCtrHelper.addPackage = function(realPath){
    try {
        if(!loadedPackages[realPath]){
            loadedPackages[realPath] = require(realPath);
        }
    } catch(e) {
        return Logger.printError("WHATAP-512", "PackageControler helper", e, false);
    }
};
PackageCtrHelper.dynamicHook = function(pkg, realPath, funcName) {
    if(!pkg || !funcName) {
        return;
    }

    var Interceptor = require('../core/interceptor').Interceptor;
    var _aop = new Interceptor(null);
    _aop.both(pkg, funcName,
        function(obj, args, lctx){
            // transaction step
            var ctx = lctx.context;
            if(!ctx) { return; }

            var msg = realPath + ' ' + funcName;
            var hash = HashUtil.hashFromString(msg);
            var step = new MethodStep();
            DataTextAgent.METHOD.add(hash, msg);
            step.hash = hash;
            step.start_time = ctx.getElapsedTime();
            pkg.__whatap_step__ = step;
        },
        function(obj, args, ret, lctx){
            var ctx = lctx.context;
            if(!ctx) { return; }
            var step = pkg.__whatap_step__;
            step.elapsed = ctx.getElapsedTime() - step.start_time;
            ctx.profile.add(step);
        });
};
PackageCtrHelper.getLoadedPackageList = function(filter){
    return loadedPackageList;
};
PackageCtrHelper.getDependencies = function () {
    try {
        var package_json = require(path.join(conf['app.root'], 'package.json'));
        if (package_json == null) { return {} };
        return package_json.dependencies || {};
    } catch (e) {
        return {};
    }
};
PackageCtrHelper.getInstalledPackageList = function(){
    try {
        var packageFolder = path.join(conf['app.root'], 'node_modules'),
            dirs = fs.readdirSync( packageFolder );

        if(dirs && dirs.length > 0) {
            dirs = dirs.filter(function(item) {
                return item.indexOf('.') != 0;
            });
        }
        return dirs;
    } catch(e) {
        Logger.printError('WHATAP-004', 'Get Installed Package List ', e, true);
    }

    return [];
};
PackageCtrHelper.getPackageDetail = function(pkg){
    try {

        var packageFolder = path.join(conf['app.root'], 'node_modules'),
            packageFile = fs.readFileSync( path.join(packageFolder, pkg, 'package.json')),

            packageJson = JSON.parse(packageFile);
        return packageJson.dependencies;
    } catch(e) {
        Logger.printError('WHATAP-005', 'Get Package Detail ', e, true);
    }
};

conf.on('hook_method_patterns', function (value) {
    var methods = value.split(',');

    methods.forEach(function (path) {

        var relative_path = path.substr(0, path.lastIndexOf('/'));
        var method = path.substr(path.lastIndexOf('/') + 1, path.length);

        while(relative_path[0] == '.') {
            relative_path = relative_path.substr(1);
        }
        var root = process.cwd();
        if(root.indexOf('/bin') >= 0) {
            root = root.substr(0, root.indexOf('/bin'));
        }
        var real_path = root + relative_path;
        PackageCtrHelper.addPackage(real_path);
        if(loadedPackages[real_path]) {
            PackageCtrHelper.dynamicHook(loadedPackages[real_path], real_path, method);
        }
    });
});

module.exports = PackageCtrHelper;