/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var os = require('os');
var PackageCtrHelper = require('./packagectr-helper'),
    RequestQueue = require('../util/request-queue'),
    PackEnum = require('../pack/packenum'),
    ParamDef = require('../net/paramdef'),
    Configure = require('../conf/configure'),
    MapValue = require('../value/map-value'),
    DateUtil = require('../util/dateutil'),
    TextValue = require('../value/text-value'),
    ResourceProfile = require('../util/resourceprofile'),
    TraceContextManager = require('../trace/trace-context-manager'),
    Logger = require('../logger'),
    StatusDetector = require('../topology/status-detector'),
    BlobValue = require('../value/blob-value');

var ControlHandler = function () {
    this.queue = new RequestQueue(100);
    this._dataPackSender = null;
}

ControlHandler.prototype.add = function (p /* Pack */) {
    this.process(p);
};

ControlHandler.prototype.process = async function (p) {
    switch (p.getPackType()) {
        case PackEnum.PARAMETER:
            await this.handle(p);
            break;
        default:
            Logger.printError('WHATAP-203', 'Unknown pack received ' + p, new Error('Unknown pack'));
    }
};

ControlHandler.prototype.handle = async function (p) {
    var self = this;
    if (!this._dataPackSender) {
        this._dataPackSender = require('../data/datapack-sender');
    }
    switch (p.id) {
        case ParamDef.CONFIGURE_GET:
            var m = Configure.readCustomConfig();
            p.setMapValue(m);
            self._dataPackSender.sendResponseHide(p.getResponse());
            break;
        case ParamDef.GET_ACTIVE_STACK:
            break;
        case ParamDef.GET_ACTIVE_TRANSACTION_DETAIL:
            try {
                var threadId = p.getNumber("thread_id");
                var profile = p.getLong("profile");
                var m = getCurrentStackDetail(threadId, profile);
                p.putValue("detail", m);

                self._dataPackSender.sendResponseHide(p.getResponse());
            } catch (e) {
                Logger.printError('WHATAP-010', 'Active Transaction Detail', e, true);
            }
            break;
        case ParamDef.GET_ACTIVE_TRANSACTION_LIST:
            try {
                var m = getActiveTxList();
                p.putValue("active", m);
                self._dataPackSender.sendResponseHide(p.getResponse());
            } catch (e) {
                Logger.printError('WHATAP-203', 'Get active transaction list ', e, true);
            }
            break;
        case ParamDef.HEAP_HISTO:

            break;
        case ParamDef.LOADED_CLASS_LIST:
            try {
                var packageList = PackageCtrHelper.getLoadedPackageList();
                var mv = new MapValue();
                var indexLv = mv.newList("index");
                var nameLv = mv.newList("name");
                var typeLv = mv.newList("type");
                var superClassLv = mv.newList("superClass");
                var interfacesLv = mv.newList("interfaces");
                var resourceLv = mv.newList("resource");

                for (var i = 0; i < packageList.length; i++) {
                    var pkg = packageList[i];
                    indexLv.addLong(i);
                    nameLv.addString(pkg);
                    typeLv.addString(pkg);
                    superClassLv.addString(pkg);
                    interfacesLv.addString(pkg);
                    resourceLv.addString(pkg);
                }
                p.putValue("classlist", mv);
            } catch (e) {
                p.put("error", e.toString());
            }
            self._dataPackSender.sendResponseHide(p.getResponse());
            break;

        case ParamDef.LOADED_CLASS_DETAIL:
            var pkgName = p.getText("classname");
            try {
                var dependencies = PackageCtrHelper.getPackageDetail(pkgName);
                var subPkgNameAndVersion = '';
                for (var k in dependencies) {
                    var dep = dependencies[k];
                    subPkgNameAndVersion += k + ": " + dep + "\n";
                }
                p.put("class", subPkgNameAndVersion);
            } catch (e) {
                p.put("error", e.toString());
            }
            break;
        case ParamDef.LOADED_CLASS_REDEFINE:
            self._dataPackSender.sendResponseHide(p.getResponse());
            break;
        case ParamDef.NODE_MODULE_DEPENDENCY:

            var mv = new MapValue();
            try {
                var dependencies = PackageCtrHelper.getDependencies();
                var keys = Object.keys(dependencies);

                for (var i = 0; i < keys.length; i++) {
                    mv.putString(keys[i], JSON.stringify(dependencies[keys[i]]));
                }
                p.putValue("module", mv);
            } catch (e) {
                Logger.printError('WHATAP-003', 'Node Module Dependency ', e, true);
            }
            self._dataPackSender.sendResponseHide(p.getResponse());
            break;
        case ParamDef.GET_ENV:

            try {
                var mv = new MapValue();
                for (var k in process.env) {
                    var v = process.env[k];
                    mv.putString(k, v);
                }
                p.putValue("env", mv);
            } catch (e) {
                p.putValue("error", e.toString());
            }
            self._dataPackSender.sendResponseHide(p.getResponse());
            break;

        case ParamDef.SYSTEM_GC:
            try {
                var total1 = os.totalmem();
                var used1 = total1 - os.freemem();
                if (global.gc) {
                    global.gc();
                } else {
                    Logger.printError('WHATAP-177', 'Garbage collection unavailable.  Pass --expose-gc '
                        + 'when launching node to enable forced garbage collection.', new Error());
                }
                var total2 = os.totalmem();
                var used2 = total2 - os.freemem();

                var mv = new MapValue();
                mv.put("before-t", total1);
                mv.put("before-u", used1);
                mv.put("after-t", total2);
                mv.put("after-u", used2);
                p.putValue("gc", m);
            } catch (e) {
                p.putValue("error", e.toString());
            }
            self._dataPackSender.sendResponseHide(p.getResponse());
            break;
        case ParamDef.SET_CONFIG:
            try {
                var mv = p.getValue('config');
                var en = mv.keys();
                var props = {};
                while(en.hasMoreElements()){
                    var key = en.nextElement();
                    var val = mv.getText(key);
                    props[key] = val
                }
                Configure.saveProperty(props);
            } catch (e) {
                p.putValue("error", e.toString())
            }
            self._dataPackSender.sendResponseHide(p.getResponse());
            break;

        case ParamDef.LOADED_CLASS_DETAIL:

            break;

        case ParamDef.OPEN_SOCKET_LIST:

            break;

        case ParamDef.RESET_STRING_SENT_MARK:
            break;
        case ParamDef.AGENT_LOG_LIST:
            var o = Logger.getLogFiles();
            if (o !== null) {
                p.putValue("files", o);
            }
            self._dataPackSender.sendResponseHide(p.getResponse());
            break;

        case ParamDef.AGENT_LOG_READ:
            var file = p.getText("file");
            var endpos = p.getNumber("pos");
            var length = p.getNumber("length"); // 최대 16k 만 가능함...
            length = Math.min(length, 8000);
            if (endpos == 0) {
                endpos += length;
            }
            Logger.read(file, endpos, length, function (before, next, text) {
                p.putLong("before", before);
                p.putLong("next", next);
                p.putString("text", text);
                self._dataPackSender.sendResponseHide(p.getResponse());
            });
            break;
        case ParamDef.GET_TOPOLOGY:
            try {
                const node = await new StatusDetector().process();
                if (node) {
                    p.putValue("node", new BlobValue(node.toBytes()));
                }
                this._dataPackSender.sendResponseHide(p.getResponse());
            } catch (e) {
                Logger.printError('WHATAP-203', 'Node Module Dependency ', e, true);
            }
            break;
    }
}

function getActiveTxList() {
    var out = new MapValue()
    var time = out.newList("time");
    var tx_hash = out.newList("tx_hash");
    var tx_name = out.newList("tx_name");
    var profile = out.newList("profile");
    var ip = out.newList("ip");

    var userid = out.newList("userid");
    var elapsed = out.newList("elapsed");
    var cputime = out.newList("cputime");
    var malloc = out.newList("malloc");
    var sqlCount = out.newList("sqlCount");

    var sqlTime = out.newList("sqlTime");
    var httpcCount = out.newList("httpcCount");
    var httpcTime = out.newList("httpcTime");
    var threadId = out.newList("threadId");
    var threadStat = out.newList("threadStat");

    var act_dbc = out.newList("act_dbc");
    var act_sql = out.newList("act_sql");
    var act_httpc = out.newList("act_httpc");


    var currentTime = DateUtil.currentTime();

    //여기서 초기화가 안된 상태인경우가 있음..
    TraceContextManager = require('../trace/trace-context-manager');
    var en = TraceContextManager.getContextEnumeration();

    while (en.hasMoreElements()) {
        var ctx = en.nextElement();
        if (ctx == null)
            continue;

        time.addLong(currentTime);
        tx_hash.addLong(ctx.service_hash);
        tx_name.addString(ctx.service_name);
        profile.addLong(ctx.txid);
        ip.addLong(ctx.remoteIp);
        userid.addLong(ctx.userid);
        elapsed.addLong(ctx.getElapsedTime());
        cputime.addLong(ResourceProfile.getCPUTime() - ctx.start_cpu);
        var m = ResourceProfile.getUsedHeapSize() - ctx.start_malloc;
        malloc.addLong(m < 0 ? 0 : m);
        sqlCount.addLong(ctx.sql_count);
        sqlTime.addLong(ctx.sql_time);
        httpcCount.addLong(ctx.httpc_count);
        httpcTime.addLong(ctx.httpc_time);
        threadId.addLong(ctx._id);
        threadStat.addLong(0);

        act_dbc.addLong(ctx.active_dbc);
        act_sql.addLong(ctx.active_sqlhash);

        act_httpc.addLong(ctx.active_httpc_hash);
    }
    return out;
}

function getCurrentStackDetail(traceId, txid) {
    var out = new MapValue();

    var ctx = TraceContextManager.getContext(traceId);
    if (ctx != null && ctx.txid.equals(txid)) {
        out.putLong("time", DateUtil.currentTime());
        out.putLong("tx_hash", ctx.service_hash);
        out.putString("tx_name", ctx.service_name);
        out.putLong("profile", ctx.txid);
        out.putLong("ip", ctx.remoteIp);
        out.putLong("userid", ctx.userToken);
        out.putLong("elapsed", ctx.getElapsedTime());
        out.putLong("sqlCount", ctx.sql_count);
        out.putLong("sqlTime", ctx.sql_time);
        out.putLong("httpcCount", ctx.httpc_count);
        out.putLong("httpcTime", ctx.httpc_time);
        out.putLong("threadId", ctx.thread_id);

        ///////////////////
        if (ctx.http_content_type != null) {
            out.putString("httpContentType", ctx.http_content_type);
        }
        ///////////////////

        if (ctx.active_sqlhash != 0) {
            out.putLong("act_dbc", ctx.active_dbc);
            out.putLong("act_sql", ctx.active_sqlhash);
        }
        if (ctx.active_httpc_hash != 0) {
            out.putLong("act_httpc", ctx.active_httpc_hash);
            out.putLong("act_httpc_host", ctx.httpc_host);
            out.putLong("act_httpc_port", ctx.httpc_port);
        }
        out.putString("method", ctx.http_method);

        if (ctx.mtid.isZero() === false) {
            out.putLong("mtid", ctx.mtid);
            out.putLong("mdepth", ctx.mdepth);
            out.putLong("mcaller", ctx.mcaller_txid);
        }

        // var se = ctx.stack;
        // var stack = out.newList("callstack");
        // for (var i = 0; i < se.length; i++) {
        //     stack.addString(se[i]);
        // }

        var stack = out.newList("callstack");
        stack.addString("Last step time: " + new Date(ctx.last_footprint_time).toString());
        stack.addString("Last step to now: " +(Date.now() - ctx.last_footprint_time) +" ms");
        stack.addString("Last step message: " + ctx.last_footprint_desc);


    }
    return out;
}
module.exports = new ControlHandler();