/**
 * Copyright 2016 the WHATAP project authors. All rights reserved.
 * Use of this source code is governed by a license that
 * can be found in the LICENSE file.
 */

var WHATAP_CONF  = process.env.WHATAP_CONF || 'whatap';

var fs = require('fs'),
    path = require('path'),
    EventEmitter = require('events').EventEmitter;

var DataInputX = require('./../io/data-inputx'),
    secu = require('./../net/security-master'),
    NodeUtil = require('./../util/nodeutil'),
    HashUtil = require('./../util/hashutil'),
    StringUtil = require('./../util/string-util'),
    Cypher = require('./../util/cypher'),
    ConfigDefault = require("./config-default"),
    LogConfigDefault = require("./log-config-default"),
    ConfSysMon = require('./conf-sys-mon'),
    MapValue = require('./../value/map-value'),
    IntKeyLinkedMap = require('./../util/intkey-linkedmap'),
    lastModifiedTime = 0;

//Object.assign(ConfigDefault, require('./config-default'));
var Configuration = function() {
    var self = this;
    EventEmitter.call(this);
    this.apply(ConfigDefault);
    this.apply(LogConfigDefault);

    this.inx = 0;
    this._customProps = {};
    this._propertyFilePath = undefined;
    this._log_ignore_set = new Set();
    this.socketChk = new IntKeyLinkedMap(500,1).setMax(500);
    // this._profile_position_sql = new Set();
    // this._profile_position_httpc = new Set();
    // this._profile_position_method = new Set();

    setInterval(function(){
        self.reload();
    }, 10000);
};

Configuration.prototype = new EventEmitter();
Configuration.prototype.constructor = Configuration;
Configuration.prototype.init = function(userOpt, cb) {
    this.reload(cb);
};

var applyEmptyConf=false;
Configuration.prototype.reload = function(cb) {
    var self = this;

    this.getPropertyFilePath( function(err, propertyFile) {
        if(err) {
            if(applyEmptyConf==false){
                applyEmptyConf=true;
                require('../logger').print("WHATAP-001", "No Configure file '"+propertyFile+"'", true);

                var defKeys = Object.keys(ConfigDefault);
                var p = {};
                defKeys.forEach(function (key) {
                    if(ConfigDefault[key]) {
                        p[key] = ConfigDefault[key];
                    }
                });
                var logDefKeys = Object.keys(LogConfigDefault);
                logDefKeys.forEach(function (key) {
                    if(LogConfigDefault[key]) {
                        p[key] = LogConfigDefault[key];
                    }
                });

                if(process.env.WHATAP_SERVER_HOST){
                    p['whatap.server.host']=process.env.WHATAP_SERVER_HOST;
                }
                if(process.env.WHATAP_LICENSE){
                    p['license']=process.env.WHATAP_LICENSE;
                }
                self.apply(p);
                updatePrivate(self);
                if(cb) cb();
            }
        } else {
            if(propertyFile == null) { return; }
            applyEmptyConf=false;
            var propStats = fs.lstatSync(propertyFile);
            if(lastModifiedTime >= propStats.mtime) { return; }
            lastModifiedTime = propStats.mtime;

            var props = fs.readFileSync(propertyFile, 'utf8');
            if(props.constructor == String){
                var rowProps = props.split('\n');
                var oldCustom = self._customProps || {};
                self._customProps = {};
                for(var i = 0 ; i < rowProps.length; i++) {
                    var p = rowProps[i].split('=');
                    if(p.length > 1) {

                        var boolValue = toBoolean(p[1]);
                        if(boolValue != null) {
                            self._customProps[p[0]] = toBoolean(p[1]);
                            continue;
                        }

                        var numberValue = toNumber(p[1]);
                        if(numberValue != null) {
                            self._customProps[p[0]] = numberValue;
                            continue;
                        }
                        self._customProps[p[0]] = p[1];
                    }
                }

                var oldKeys = Object.keys(oldCustom);
                var newKeys = Object.keys(self._customProps);

                oldKeys.forEach(function (key) {
                    if(newKeys.indexOf(key) < 0) {
                        if(ConfigDefault[key] !== null || LogConfigDefault[key] !== null) {
                            self[key] = ConfigDefault[key] !== null ? ConfigDefault[key] : LogConfigDefault[key];
                        } else {
                            delete self[key];
                        }
                    }
                });
                require('../logger').print("WHATAP-203", "Config file reloaded", false);
            }

            if(!self._customProps['license'] && process.env.WHATAP_LICENSE)
                self._customProps['license'] = process.env.WHATAP_LICENSE;
            if(!self._customProps['whatap.server.host'] && process.env.WHATAP_SERVER_HOST)
                self._customProps['whatap.server.host'] = process.env.WHATAP_SERVER_HOST;
            self.apply( self._customProps );
            updatePrivate(self);
            if(cb) cb();
        }
    });

};
function updatePrivate(self){
    if(self.trace_user_using_ip) {
        self.trace_user_using_type = 1;
    } else {
        self.trace_user_using_type = 3;
    }

    self._log_ignore_set = self.getStringSet('log_ignore_set', null, ',');
    // self._profile_position_sql = self.getStringSet('profile_position_sql', null, ',');
    // self._profile_position_httpc = self.getStringSet('profile_position_httpc', null, ',');
    // self._profile_position_method = self.getStringSet('profile_position_method', null, ',');
}

Configuration.prototype.apply = function(properties) {
    for(var k in properties) {
        this.setProperty(k, properties[k]);
    }

    var set = this.getStringSet("whatap.server.host", "127.0.0.1", "/");
    var ips = Array.from(set);
    this.whatap_server_host = ips;

    if( this.getProperty('whatap_micro_enabled', false) === true || process.env.WHATAP_MICRO_ENABLED === "true"){
        this.kubeNodeName(properties);
    }

    ConfSysMon.apply(properties);
};

Configuration.prototype.kubeNodeName = function(properties){
    var POD_NAME = process.env['POD_NAME'] || process.env['PODNAME'];
    this.OKIND_NAME = this.getProperty('whatap.okind');
    if(!this.OKIND_NAME || this.OKIND_NAME.length < 1){
        this.OKIND_NAME = POD_NAME ? cutOut(POD_NAME, '-') : '';
        this.setProperty('whatap.okind', this.OKIND_NAME);
    }
    this.OKIND = StringUtil.isEmpty(this.OKIND_NAME) ? 0 : HashUtil.hash(this.OKIND_NAME);
    secu.OKIND = this.OKIND;
    secu.OKIND_NAME = this.OKIND_NAME;

    this.ONODE_NAME = this.getProperty("whatap.onode");
    if (!this.ONODE_NAME || (this.ONODE_NAME && this.ONODE_NAME.length < 1)) {
        this.ONODE_NAME = process.env['NODE_NAME'] || process.env['WHATAP_ONODE'];
    }
    if (!this.ONODE_NAME || (this.ONODE_NAME && this.ONODE_NAME.length < 1)) {
        this.ONODE_NAME = process.env["NODE_IP"];
    }
    this.ONODE = StringUtil.isEmpty(this.ONODE_NAME) ? 0 : HashUtil.hashFromString(this.ONODE_NAME);
    secu.ONODE = this.ONODE;
    secu.ONODE_NAME = this.ONODE_NAME;
}

Configuration.prototype.getPropertyFilePath = function(cb) {
    var self = this;
    if(self._propertyFilePath) {
        if(cb) {
            cb(null, self._propertyFilePath);
        }
    }

    var confDir= '.';
    if(process.env.WHATAP_CONF_DIR){
        confDir=  process.env.WHATAP_CONF_DIR;
    }else{
        confDir =this.getProperty('app.root', process.cwd());
    }

    var default_conf = WHATAP_CONF+".conf";
    var configFile = this.getProperty('whatap.config', default_conf);

    var confFullPathFile=path.join(confDir, configFile);
    fs.access(confFullPathFile, function (err) {
        if (err) {
            self._propertyFilePath = null;
            if (cb) cb(err, confFullPathFile);
        } else {
            self._propertyFilePath = confFullPathFile;
            if (cb) cb(null, confFullPathFile);
        }
    });
};
Configuration.prototype.getCustomProps = function(){
    var props = {};
    for(var k in this._customProps){
        props[k] = this._customProps[k];
    }
    return props;
};
Configuration.prototype.saveProperty = function(keyValues){
    var self = this;
    this.getPropertyFilePath(function(err, propertyFile){
        for(var k in keyValues){
            var v = keyValues[k];
            if(!v || v.length < 1){
                delete self._customProps[k];
            }else{
                self._customProps[k] = v;
            }
        }

        if(!self._customProps['whatap.server.host'] && process.env.WHATAP_SERVER_HOST){
            self._customProps['whatap.server.host']=process.env.WHATAP_SERVER_HOST;
        }
        if(!self._customProps['license'] && process.env.WHATAP_LICENSE){
            self._customProps['license']=process.env.WHATAP_LICENSE;
        }
        var writeContents = '';
        for(var k in self._customProps){
            var v = self._customProps[k];
            writeContents += k +'=' + v +'\n';
        }
        fs.writeFile(propertyFile, writeContents , 'utf-8', function(err, data){
            if(err){
                return console.error(err);
            }
        })
        for(var k in self._customProps){
            var v = self._customProps[k];
            self.setProperty(k, v);
        }
    });
};
Configuration.prototype.setProperty = function(key, value) {
    if(this[key] == value) { return; }
    this[key] = value;
    this.emit(key, value);
};
Configuration.prototype.getProperty = function(key, defaultValue) {
    if(this[key] === undefined){
        return defaultValue;
    } else{
        return this[key];
    }
};

Configuration.prototype.getWhatapServerHost = function () {
    var _len = this.whatap_server_host.length;
    switch (_len) {
        case 0:
            return '127.0.0.1';
        case 1:
            return this.whatap_server_host[0];
    }
    var _inx = this.inx;
    var out = this.whatap_server_host[_inx];
    this.inx = (_inx + 1) % _len;
    return out;
};
Configuration.prototype.updateNetCypherKey = function(data /* byte[] */) {
    try{
        var data = secu.cypher.decrypt(data);
        var inD = new DataInputX(data);
        this.TRANSFER_KEY = inD.readInt();
        this.SECURE_KEY = inD.readBlob();
        this.HIDE_KEY = inD.readInt();
        this.cypher = new Cypher(this.SECURE_KEY, this.HIDE_KEY);
    } catch(e) {

    }
};
Configuration.prototype.readCustomConfig = function(){
    var props = this.getCustomProps();
    var mv = new MapValue();
    for(var k in props){
        mv.putString(k, props[k]);
    }
    return mv;
};
Configuration.prototype.isIgnoreLog = function (id) {
    return this._log_ignore_set.has(id);
};
Configuration.prototype.setSocketChk = function (socket) {
    if(socket.whatapId == undefined) {
        socket.whatapId = (new Date()).getTime().toString(36) + Math.random().toString(36).slice(2);
        //require('../logger').print("WHATAP-TEST", 'Socket whatapId : ' + socket.whatapId, false);
    }
    //else {
    //    require('../logger').print("WHATAP-TEST", '[REUSE] Socket whatapId : ' + socket.whatapId, false);
    //}
    if(this.socketChk.get(socket.whatapId) == null) {
        this.socketChk.put(socket.whatapId , socket.whatapId);
        return false;
    } else return true;
}
Configuration.prototype.getStringSet = function (key, defaultValue, deli) {
    var set = new Set();
    var value = this[key] || defaultValue;
    if(value == null) { return set; }

    var arr = value.split(deli);
    for(var i=0; i<arr.length; i++) {
        var x = arr[i].trim();
        if(x)set.add(replaceCarriageReturns(x));
    }
    return set;
};

Configuration.prototype.getLogSinkProperty = function (id, key) {
    return this.getProperty("pii_mask_logsink_"+id + "_" + key);
}

Configuration.prototype.getSqlProperty = function (id, key) {
    return this.getProperty("pii_mask_sql_"+id + "_" + key)
}

function toNumber(value) {
    if(typeof value == 'number') {
        return value;
    }

    var number = Number(value);
    if(isNaN(number)) {
        return null;
    } else {
        return number;
    }
}
function toBoolean(value) {
    if(typeof value == 'boolean') {
        return value;
    }

    if(typeof value == 'string') {
        if(value == 'true') {
            return true;
        } else if(value == 'false') {
            return false;
        }
    }
    return null;
}

function replaceCarriageReturns(str) {
    let regxp = /\r/g;
    str = str.replace(regxp, "");
    return str;
}

function cutOut(val, delim) {
    if (val === undefined)
        return val;
    try {
        var x = val.lastIndexOf(delim);
        if (x <= 0)
            return;
        return val.substring(0, x);
    } catch (e) {
        return val;
    }
}

const conf = new Configuration();

conf.on('trace_ignore_url_set', function(props){
    if(!props || props.constructor !== String){
        return;
    }

    var ignoreUrls = props.split(',');
    conf._trace_ignore_url_set = {};
    for(var i = 0; i < ignoreUrls.length; i++){
        var ignoreUrl = ignoreUrls[i];
        var serviceHash = HashUtil.hashFromString(ignoreUrl);
        conf._trace_ignore_url_set[serviceHash] = true;
    }
});

conf.on('trace_ignore_url_prefix', function(props){
    if(props && props.constructor === String && props.length > 0 ){
        conf._is_trace_ignore_url_prefix = true;
    }else{
        conf._is_trace_ignore_url_prefix = false;
    }
});

conf.on('trace_ignore_err_cls_contains', function(props){
    if(props && props.constructor === String && props.length > 0 ){
        conf._is_trace_ignore_err_cls_contains = true;
    }else{
        conf._is_trace_ignore_err_cls_contains = false;
    }
});

conf.on('trace_ignore_err_msg_contains', function(props){
    if(props && props.constructor === String && props.length > 0 ){
        conf._is_trace_ignore_err_msg_contains = true;
    }else{
        conf._is_trace_ignore_err_msg_contains = false;
    }
});


conf.on('mtrace_spec', function(props){
    if (!props || props.length == 0) {
        conf.mtrace_spec_hash = 0;
    } else {
        props = props.replace(',', '_');
        conf.mtrace_spec_hash = HashUtil.hash(props);
    }
})

module.exports = conf;